%
% file: test_CAMS_Africa.m
%
% Examples of reading a subset of CAMS Radiation volume over Africa (Jade)
%
% This file contains ONE TEST PER SECTION. You can run each test with "Evaluate current section" command (Ctrl+Enter).
%
% authors - date: Laurent Saboret, Transvalor - 2017
%

%% Display 15 min Gc and GHI maps

clear all; close all; clc

% Get folder containing CAMS volume over Africa.
% You may either change the next line or select the folder when prompted.
CAMS_DIR = '/mnt/data/CAMS-Africa';
if ~exist(CAMS_DIR, 'dir')
    CAMS_DIR = uigetdir('', 'Select the folder containing CAMS volume over Africa')
end

% CAMS Radiation volume bounding box in degrees
lat_min = -37.1;
lat_max =  40.1;
lon_min = -20.1;
lon_max =  54.1;
grid_step = 0.2;

% Compute CAMS Radiation volume size in pixels
width  = round((lon_max - lon_min) / grid_step); % 371
height = round((lat_max - lat_min) / grid_step); % 386

% Test bounding box in pixels (1-based).
% Feel free to change these values.
left=1
right=width
top=1
bottom=height

% Test time step (end of interval).
% Feel free to change these values.
% Note: Days 2005-01-11 and 2005-01-12 are missing in Apollo clouds
year=2005
month=4
day=1
hr=12
mn=00
%
% convert test frame to julian day
jd = date2jd(year, month, day, hr, mn, 0);

% CAMS Radiation volume time start
start_jd = date2jd(year, 1, 1, 0, 15, 0); % January 1st of year 'year' at 0h15

% Read Gc map in Wh/m2
filename = sprintf('%s/gc_%d.h5', CAMS_DIR, year);
time = (jd - start_jd)*96 + 1; % start time index
gc_map = h5read(filename, '/set', [left top time], [width height 1]);
gc_map = gc_map';
gc_map = double(gc_map);
gc_map(253<gc_map)=nan; % remove missing and invalid values
% gc_map(gc_map == 255)=nan; % remove missing values
gc_map = gc_map * 1.2; % convert to Wh/m2

% Display Gc map
figure
set(gcf,'position',[100 100 700 600]) % left, bottom, width, height
% imagesc(gc_map, [-100, 300]); % use color table that enhances missing and invalid values
imagesc(gc_map); % use automatic color table
fig_title=sprintf('Gc 15 min (Wh/m2) on %04d-%02d-%02d at %02d:%02d UT', year, month, day, hr, mn);
title(fig_title);
colorbar

% Read GHI map in Wh/m2
filename = sprintf('%s/ghi_%d.h5', CAMS_DIR, year);
time = (jd - start_jd)*96 + 1; % start time index
ghi_map = h5read(filename, '/set', [left top time], [width height 1]);
ghi_map=ghi_map';
ghi_map = double(ghi_map);
ghi_map(253<ghi_map)=nan; % remove missing and invalid values
% ghi_map(ghi_map == 255)=nan; % remove missing values
ghi_map = ghi_map * 1.2; % convert to Wh/m2

% Display GHI map
figure
set(gcf,'position',[200 200 700 600]) % left, bottom, width, height
% imagesc(ghi_map, [-100, 300]); % use color table that enhances missing and invalid values
imagesc(ghi_map); % use automatic color table
fig_title=sprintf('GHI 15 min (Wh/m2) on %04d-%02d-%02d at %02d:%02d UT', year, month, day, hr, mn);
title(fig_title);
colorbar


%% Display 15 min GHI as a video 

clear all; close all; clc

% Get folder containing CAMS volume over Africa.
% You may either change the next line or select the folder when prompted.
CAMS_DIR = '/mnt/data/CAMS-Africa';
if ~exist(CAMS_DIR, 'dir')
    CAMS_DIR = uigetdir('', 'Select the folder containing CAMS volume over Africa')
end

% CAMS Radiation volume bounding box in degrees
lat_min = -37.1;
lat_max =  40.1;
lon_min = -20.1;
lon_max =  54.1;
grid_step = 0.2;

% Compute CAMS Radiation volume size in pixels
width  = round((lon_max - lon_min) / grid_step); % 371
height = round((lat_max - lat_min) / grid_step); % 386

% Test bounding box in pixels (1-based).
% Feel free to change these values.
left=1
right=width
top=1
bottom=height

% Test day.
% Feel free to change these values.
% Note: Days 2005-01-11 and 2005-01-12 are missing in Apollo clouds.
year=2005
month=4
day=1
%
% convert test day to julian day
jd = date2jd(year, month, day, 0, 15, 0); % test day at 0h15

% CAMS Radiation volume time start
start_jd = date2jd(year, 1, 1, 0, 15, 0); % January 1st of year 'year' at 0h15

% Read GHI map in Wh/m2
filename = sprintf('%s/ghi_%d.h5', CAMS_DIR, year);
time = (jd - start_jd)*96 + 1; % start time index
ghi_map = h5read(filename, '/set', [left top time], [width height 96]);
ghi_map = double(ghi_map);
ghi_map(253<ghi_map)=nan; % remove missing and invalid values
% ghi_map(ghi_map == 255)=nan; % remove missing values
ghi_map = ghi_map * 1.2; % convert to Wh/m2

% Display GHI map as a video
figure
set(gcf,'position',[200 200 700 700]) % left, bottom, width, height
%vc(permute(ghi_map, [2 1 3]), [-100, 300]) % use color table that enhances missing and invalid values
vc(permute(ghi_map, [2 1 3])) % use automatic color table


%% Plot 15 min GHI time series

clear all; close all; clc

% Get folder containing CAMS volume over Africa.
% You may either change the next line or select the folder when prompted.
CAMS_DIR = '/mnt/data/CAMS-Africa';
if ~exist(CAMS_DIR, 'dir')
    CAMS_DIR = uigetdir('', 'Select the folder containing CAMS volume over Africa')
end

% CAMS Radiation volume bounding box in degrees
lat_min = -37.1;
lat_max =  40.1;
lon_min = -20.1;
lon_max =  54.1;
grid_step = 0.2;

% Compute CAMS Radiation volume size in pixels
width  = round((lon_max - lon_min) / grid_step); % 371
height = round((lat_max - lat_min) / grid_step); % 386

% Test location.
% Feel free to select a location in Africa.
% location='Kharthoum'; % location name
% lat=15.5030;
% lon=32.5604;
%
location='Tangier'; % location name
lat=35.63;
lon=-5.7;

% Move to center of CAMS Radiation volume pixel
x = round((lon - lon_min)/grid_step + 0.5);
y = round((lat_max - lat)/grid_step + 0.5);
lat = lat_max - (y-0.5)*grid_step
lon = lon_min + (x-0.5)*grid_step
h = -999; % automatically computed from SRTM elevation database

% Test period.
% Feel free to change these values.
% Note: Days 2005-01-11 and 2005-01-12 are missing in Apollo clouds.
year1=2005; month1=3; day1=1;
year2=2005; month2=3; day2=10;
assert(year1 == year2, 'This example is limited to a single year');
%
yyyymmdd1 = sprintf('%04d-%02d-%02d', year1, month1, day1); % compute test period as YYYY-MM-DD
yyyymmdd2 = sprintf('%04d-%02d-%02d', year2, month2, day2);
%
jd1 = date2jd(year1, month1, day1,  0, 15, 0); % get first and last steps as julian days
jd2 = date2jd(year2, month2, day2, 24,  0, 0);

% CAMS Radiation volume time step
duration=15; % min
dt=duration/60.0; % irradiation/irradiance factor = 0.25

% CAMS Radiation volume time start
start_jd = date2jd(year1, 1, 1, 0, 15, 0); % January 1st of year 'year1' at 0h15

% Read GHI time series in Wh/m2
filename = sprintf('%s/ghi_%d.h5', CAMS_DIR, year1);
time = (jd1 - start_jd)*96 + 1; % start time index
nb_steps = round(jd2 - jd1)*96; % number of steps in period
ghi_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
ghi_time_series = double(ghi_time_series);
ghi_time_series(253 < ghi_time_series)=nan; % remove missing and invalid values
ghi_time_series = ghi_time_series * 1.2; % convert to Wh/m2

% Plot GHI
figure
set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
jds = [jd1:1/96:jd2]; % list of time steps as julian days
[years, months, days, hours, minutes, seconds] = jd2date(jds); % list of time steps as gregorian dates
datenums = datenum([years' months' days' hours' minutes' seconds']); % list of time steps as datenums
plot(datenums, ghi_time_series(:))
ticks = [round(datenums(1)):round(datenums(end))]; % 1 tick per day
set(gca, 'XTick', ticks);
datetick('x','yyyy/mm/dd','keepticks','keeplimits');
fig_title=sprintf('GHI 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
title(fig_title);


%% Plot CAMS HDF5 volume vs CAMS Radiation web service

clear all; close all; clc

% TODO Please enter your email below (requires an account on www.soda-pro.com)
email_address = 'laurent.saboret@transvalor.com' ;   % email SoDa

% Get folder containing CAMS volume over Africa.
% You may either change the next line or select the folder when prompted.
CAMS_DIR = '/mnt/data/CAMS-Africa';
if ~exist(CAMS_DIR, 'dir')
    CAMS_DIR = uigetdir('', 'Select the folder containing CAMS volume over Africa')
end

% CAMS Radiation volume bounding box in degrees
lat_min = -37.1;
lat_max =  40.1;
lon_min = -20.1;
lon_max =  54.1;
grid_step = 0.2;

% Compute CAMS Radiation volume size in pixels
width  = round((lon_max - lon_min) / grid_step); % 371
height = round((lat_max - lat_min) / grid_step); % 386

% Test location.
% Feel free to select a location in Africa.
% location='Kharthoum'; % location name
% lat=15.5030;
% lon=32.5604;
%
location='Tangier'; % location name
lat=35.63;
lon=-5.7;

% Move to center of CAMS Radiation volume pixel
x = round((lon - lon_min)/grid_step + 0.5);
y = round((lat_max - lat)/grid_step + 0.5);
lat = lat_max - (y-0.5)*grid_step
lon = lon_min + (x-0.5)*grid_step
h = -999; % automatically computed from SRTM elevation database

% Test period.
% Feel free to change these values.
% Note: Days 2005-01-11 and 2005-01-12 are missing in Apollo clouds.
year1=2005; month1=3; day1=1;
year2=2005; month2=3; day2=10;
assert(year1 == year2, 'This example is limited to a single year');
%
yyyymmdd1 = sprintf('%04d-%02d-%02d', year1, month1, day1); % compute test period as YYYY-MM-DD
yyyymmdd2 = sprintf('%04d-%02d-%02d', year2, month2, day2);
%
ddd1 = dayofyear(year1, month1, day1); % compute days of year
ddd2 = dayofyear(year2, month2, day2);
%
jd1 = date2jd(year1, month1, day1,  0, 15, 0); % get first and last steps as julian days
jd2 = date2jd(year2, month2, day2, 24,  0, 0);

% CAMS Radiation volume time step
duration=15; % min
dt=duration/60.0; % irradiation/irradiance factor = 0.25

% CAMS Radiation volume time start
start_jd = date2jd(year1, 1, 1, 0, 15, 0); % January 1st of year 'year1' at 0h15

% Run CAMS Radiation request
CAMS_RAD = cams_radiation_wget(email_address, [lat lon h], yyyymmdd1, yyyymmdd2, duration);
fprintf('\n');

% % Debug: round irradiations like HDF5 volumes
% CAMS_RAD.CLS_GHI = round(CAMS_RAD.CLS_GHI/1.2)*1.2;
% CAMS_RAD.CLS_BHI = round(CAMS_RAD.CLS_BHI/1.2)*1.2;
% CAMS_RAD.CLS_DHI = round(CAMS_RAD.CLS_DHI/1.2)*1.2;
% CAMS_RAD.CLS_BNI = round(CAMS_RAD.CLS_BNI/1.2)*1.2;
% CAMS_RAD.GHI = round(CAMS_RAD.GHI/1.2)*1.2;
% CAMS_RAD.BHI = round(CAMS_RAD.BHI/1.2)*1.2;
% CAMS_RAD.DHI = round(CAMS_RAD.DHI/1.2)*1.2;
% CAMS_RAD.BNI = round(CAMS_RAD.BNI/1.2)*1.2;

filename = sprintf('%s/gc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read gc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1; % start time index
    nb_steps = round(jd2 - jd1)*96; % number of steps in period
    gc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    gc_time_series = double(gc_time_series);
    gc_time_series(253 < gc_time_series)=nan; % remove missing and invalid values
    gc_time_series = gc_time_series * 1.2; % convert to Wh/m2
    
    % Plot gc
    figure
    fig_title=sprintf('gc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'gc CAMS Radiation request', CAMS_RAD.CLS_GHI, ...
                  'gc HDF5 volume', gc_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1; % start time index
    nb_steps = round(jd2 - jd1)*96; % number of steps in period
    bc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bc_time_series = double(bc_time_series);
    bc_time_series(253 < bc_time_series)=nan; % remove missing and invalid values
    bc_time_series = bc_time_series * 1.2; % convert to Wh/m2
    
    % Plot bc
    figure
    fig_title=sprintf('bc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'bc CAMS Radiation request', CAMS_RAD.CLS_BHI, ...
                  'bc HDF5 volume', bc_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/dc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read dc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    dc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    dc_time_series = double(dc_time_series);
    dc_time_series(253 < dc_time_series)=nan; % remove missing and invalid values
    dc_time_series = dc_time_series * 1.2; % convert to Wh/m2
    
    % Plot dc
    figure
    fig_title=sprintf('dc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'dc CAMS Radiation request', CAMS_RAD.CLS_DHI, ...
                  'dc HDF5 volume', dc_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bnc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bnc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    bnc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bnc_time_series = double(bnc_time_series);
    bnc_time_series(253 < bnc_time_series)=nan; % remove missing and invalid values
    bnc_time_series = bnc_time_series * 1.2; % convert to Wh/m2

    % Plot bnc
    figure
    fig_title=sprintf('bnc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'bnc CAMS Radiation request', CAMS_RAD.CLS_BNI, ...
                  'bnc HDF5 volume', bnc_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/ghi_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read ghi time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    ghi_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    ghi_time_series = double(ghi_time_series);
    ghi_time_series(253 < ghi_time_series)=nan; % remove missing and invalid values
    ghi_time_series = ghi_time_series * 1.2; % convert to Wh/m2
    
    % Plot ghi
    figure
    fig_title=sprintf('ghi 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'ghi CAMS Radiation request', CAMS_RAD.GHI, ...
                  'ghi HDF5 volume', ghi_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bhi_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bhi time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    bhi_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bhi_time_series = double(bhi_time_series);
    bhi_time_series(253 < bhi_time_series)=nan; % remove missing and invalid values
    bhi_time_series = bhi_time_series * 1.2; % convert to Wh/m2
    
    % Plot bhi
    figure
    fig_title=sprintf('bhi 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'bhi CAMS Radiation request', CAMS_RAD.BHI, ...
                  'bhi HDF5 volume', bhi_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/dhi_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read dhi time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    dhi_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    dhi_time_series = double(dhi_time_series);
    dhi_time_series(253 < dhi_time_series)=nan; % remove missing and invalid values
    dhi_time_series = dhi_time_series * 1.2; % convert to Wh/m2
    
    % Plot dhi
    figure
    fig_title=sprintf('dhi 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'dhi CAMS Radiation request', CAMS_RAD.DHI, ...
                  'dhi HDF5 volume', dhi_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bni_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bni time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    bni_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bni_time_series = double(bni_time_series);
    bni_time_series(253 < bni_time_series)=nan; % remove missing and invalid values
    bni_time_series = bni_time_series * 1.2; % convert to Wh/m2

    % Plot bni
    figure
    fig_title=sprintf('bni 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    irrad_compare(fig_title, 'yyyy-mm-dd', CAMS_RAD.ymdhms, ...
                  'bni CAMS Radiation request', CAMS_RAD.BNI, ...
                  'bni HDF5 volume', bni_time_series, ...
                  true, false); % plot = true, stats = false
    set(gcf,'position',[100 100 1200 600]) % left, bottom, width, height
else
    fprintf('Cannot open %s. Skip it\n', filename);
end


%% Compare CAMS HDF5 volume vs CAMS Radiation web service using scatter plots

clear all; close all; clc

% TODO Please enter your email below (requires an account on www.soda-pro.com)
email_address = 'laurent.saboret@transvalor.com' ;   % email SoDa

% Get folder containing CAMS volume over Africa.
% You may either change the next line or select the folder when prompted.
CAMS_DIR = '/mnt/data/CAMS-Africa';
if ~exist(CAMS_DIR, 'dir')
    CAMS_DIR = uigetdir('', 'Select the folder containing CAMS volume over Africa')
end

% CAMS Radiation volume bounding box in degrees
lat_min = -37.1;
lat_max =  40.1;
lon_min = -20.1;
lon_max =  54.1;
grid_step = 0.2;

% Compute CAMS Radiation volume size in pixels
width  = round((lon_max - lon_min) / grid_step); % 371
height = round((lat_max - lat_min) / grid_step); % 386

% Test location.
% Feel free to select a location in Africa.
% location='Kharthoum'; % location name
% lat=15.5030;
% lon=32.5604;
%
location='Tangier'; % location name
lat=35.63;
lon=-5.7;

% Move to center of CAMS Radiation volume pixel
x = round((lon - lon_min)/grid_step + 0.5);
y = round((lat_max - lat)/grid_step + 0.5);
lat = lat_max - (y-0.5)*grid_step
lon = lon_min + (x-0.5)*grid_step
h = -999; % automatically computed from SRTM elevation database

% Test period.
% Feel free to change these values.
% Note: Days 2005-01-11 and 2005-01-12 are missing in Apollo clouds.
year1=2005; month1=01; day1=01;
year2=2005; month2=12; day2=31;
assert(year1 == year2, 'This example is limited to a single year');
%
yyyymmdd1 = sprintf('%04d-%02d-%02d', year1, month1, day1); % compute test period as YYYY-MM-DD
yyyymmdd2 = sprintf('%04d-%02d-%02d', year2, month2, day2);
%
ddd1 = dayofyear(year1, month1, day1); % compute days of year
ddd2 = dayofyear(year2, month2, day2);
%
jd1 = date2jd(year1, month1, day1,  0, 15, 0); % get first and last steps as julian days
jd2 = date2jd(year2, month2, day2, 24,  0, 0);

% CAMS Radiation volume time step
duration=15; % min
dt=duration/60.0; % irradiation/irradiance factor = 0.25

% CAMS Radiation volume time start
start_jd = date2jd(year1, 1, 1, 0, 15, 0); % January 1st of year 'year1' at 0h15

% Run CAMS Radiation request
CAMS_RAD = cams_radiation_wget(email_address, [lat lon h], yyyymmdd1, yyyymmdd2, duration);
fprintf('\n');

% % Debug: round irradiations like HDF5 volumes
% CAMS_RAD.CLS_GHI = round(CAMS_RAD.CLS_GHI/1.2)*1.2;
% CAMS_RAD.CLS_BHI = round(CAMS_RAD.CLS_BHI/1.2)*1.2;
% CAMS_RAD.CLS_DHI = round(CAMS_RAD.CLS_DHI/1.2)*1.2;
% CAMS_RAD.CLS_BNI = round(CAMS_RAD.CLS_BNI/1.2)*1.2;
% CAMS_RAD.GHI = round(CAMS_RAD.GHI/1.2)*1.2;
% CAMS_RAD.BHI = round(CAMS_RAD.BHI/1.2)*1.2;
% CAMS_RAD.DHI = round(CAMS_RAD.DHI/1.2)*1.2;
% CAMS_RAD.BNI = round(CAMS_RAD.BNI/1.2)*1.2;

filename = sprintf('%s/gc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read gc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1; % start time index
    nb_steps = round(jd2 - jd1)*96; % number of steps in period
    gc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    gc_time_series = double(gc_time_series);
    gc_time_series(253 < gc_time_series)=nan; % remove missing and invalid values
    gc_time_series = gc_time_series * 1.2; % convert to Wh/m2
    
    % Draw gc scatter plot       
    figure
    fig_title=sprintf('gc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(gc_time_series, CAMS_RAD.CLS_GHI, [], 1, 1, 'Wh/m2', ...
                           {'gc HDF5 volume' 'gc CAMS Radiation request'});
    title(fig_title);
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    bc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bc_time_series = double(bc_time_series);
    bc_time_series(253 < bc_time_series)=nan; % remove missing and invalid values
    bc_time_series = bc_time_series * 1.2; % convert to Wh/m2
    
    % Draw bc scatter plot       
    figure
    fig_title=sprintf('bc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(bc_time_series, CAMS_RAD.CLS_BHI, [], 1, 1, 'Wh/m2', ...
                           {'bc HDF5 volume' 'bc CAMS Radiation request'});
    title(fig_title);
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/dc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read dc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    dc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    dc_time_series = double(dc_time_series);
    dc_time_series(253 < dc_time_series)=nan; % remove missing and invalid values
    dc_time_series = dc_time_series * 1.2; % convert to Wh/m2
    
    % Draw dc scatter plot       
    figure
    fig_title=sprintf('dc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(dc_time_series, CAMS_RAD.CLS_DHI, [], 1, 1, 'Wh/m2', ...
                           {'dc HDF5 volume' 'dc CAMS Radiation request'});
    title(fig_title);
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bnc_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bnc time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    bnc_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bnc_time_series = double(bnc_time_series);
    bnc_time_series(253 < bnc_time_series)=nan; % remove missing and invalid values
    bnc_time_series = bnc_time_series * 1.2; % convert to Wh/m2
    
    % Draw bnc scatter plot       
    figure
    fig_title=sprintf('bnc 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(bnc_time_series, CAMS_RAD.CLS_BNI, [], 1, 1, 'Wh/m2', ...
                           {'bnc HDF5 volume' 'bnc CAMS Radiation request'});
    title(fig_title);
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/ghi_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read ghi time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    ghi_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    ghi_time_series = double(ghi_time_series);
    ghi_time_series(253 < ghi_time_series)=nan; % remove missing and invalid values
    ghi_time_series = ghi_time_series * 1.2; % convert to Wh/m2
    
    % Draw ghi scatter plot       
    figure
    fig_title=sprintf('ghi 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(ghi_time_series, CAMS_RAD.GHI, [], 1, 1, 'Wh/m2', ...
                           {'ghi HDF5 volume' 'ghi CAMS Radiation request'});
    title(fig_title);
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bhi_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bhi time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    bhi_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bhi_time_series = double(bhi_time_series);
    bhi_time_series(253 < bhi_time_series)=nan; % remove missing and invalid values
    bhi_time_series = bhi_time_series * 1.2; % convert to Wh/m2
    
    % Draw bhi scatter plot       
    figure
    fig_title=sprintf('bhi 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(bhi_time_series, CAMS_RAD.BHI, [], 1, 1, 'Wh/m2', ...
                           {'bhi HDF5 volume' 'bhi CAMS Radiation request'});
    title(fig_title);
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/dhi_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read dhi time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    dhi_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    dhi_time_series = double(dhi_time_series);
    dhi_time_series(253 < dhi_time_series)=nan; % remove missing and invalid values
    dhi_time_series = dhi_time_series * 1.2; % convert to Wh/m2
    
    % Draw dhi scatter plot       
    figure
    fig_title=sprintf('dhi 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(dhi_time_series, CAMS_RAD.DHI, [], 1, 1, 'Wh/m2', ...
                           {'dhi HDF5 volume' 'dhi CAMS Radiation request'});
    title(fig_title); 
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

filename = sprintf('%s/bni_%d.h5', CAMS_DIR, year1);
if exist(filename)
    % Read bni time series in Wh/m2
    time = (jd1 - start_jd)*96 + 1;
    nb_steps = round(jd2 - jd1)*96;
    bni_time_series = h5read(filename, '/set', [x y time], [1 1 nb_steps]);
    bni_time_series = double(bni_time_series);
    bni_time_series(253 < bni_time_series)=nan; % remove missing and invalid values
    bni_time_series = bni_time_series * 1.2; % convert to Wh/m2

    % Draw bni scatter plot       
    figure
    fig_title=sprintf('bni 15 min (Wh/m2) at %s (%.3f, %.3f) from %s to %s', location, lat, lon, yyyymmdd1, yyyymmdd2);
    stats = statd_irrad_v6(bni_time_series, CAMS_RAD.BNI, [], 1, 1, 'Wh/m2', ...
                           {'bni HDF5 volume' 'bni CAMS Radiation request'});
    title(fig_title);
else
    fprintf('Cannot open %s. Skip it\n', filename);
end

