function HC4 = cams_radiation_wget(email_address, geopoint, start_date, end_date, duration, verbose_output, time, server)
% HC4 = cams_radiation_wget(email_address, geopoint, start_date, end_date, duration, verbose_output, time, server)
%
% Description:
% Use CAMS Radiation WPS to retrieve IRRADIATION (Wh/m2) from SoDa distant server
%
% Inputs: 
% email_address  : email, 'myname@mydomain.com' (must be subscribed to www.soda-pro.com)
% geopoint       : [lat lon] or [lat lon h] in deg/deg/m
% start_date     : start date, format 'YYYY-MM-DD'
% end_date       : end date, format 'YYYY-MM-DD'
% duration       : time step: month 'm', day 'd', hour 'h', 15 or 1 minutes
% verbose_output : boolean to retrieve other information such as aerosols; only applicable to 'UT' and 1-min
% time           : time reference 'UT' or 'TST'
% server         : SoDa server. Default is 'http://www.soda-is.com'.
%
% Outputs:
% HC4 : structure containing location, period, IRRADIATION (in Wh/m2), etc.
%
% Example:
% HC4 = cams_radiation_wget('myname@mydomain.com', [45.0 25.0], '2008-08-01', '2008-08-03', 15)

HC4 = [];

% default values
if (nargin < 6)
    verbose_output = 0;
end
if (nargin < 7)
    time = 'UT';
end
if (nargin < 8)
    server = 'http://www.soda-is.com';
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Call CAMS Radiation WPS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Convert dates to format [yyyy doy]
tmp = split(start_date,'-');
ymd_begin = [str2double(tmp{1}) str2double(tmp{2}) str2double(tmp{3})];
tmp = split(end_date,'-');
ymd_end = [str2double(tmp{1}) str2double(tmp{2}) str2double(tmp{3})];

% if elevation = -999 or is missing, SRTM elevation will be used
if (size(geopoint,2)==2) || (size(geopoint,2)==3 && geopoint(3) < -900)
    geopoint(:,3) = -999;
end

% CAMS Radiation WPS expects time reference = UT or TST
switch(time)
    case 'TST'
        ; % nothing to do
    case {'TU', 'UT'}
        time = 'UT';
    otherwise
        error('Incorrect time reference');
end

% parse duration and check verbose_output
switch(duration)
    case 1 % min
        intraday = 1;
        dt = duration/60; % duration in hours
        wps_duration='PT01M';
        if ~strcmp(time, 'UT')
            verbose_output=0; % -v option is supported if 1-minute UT time series only
        end
    case 15 % min
        intraday = 1;
        dt = duration/60;
        wps_duration='PT15M';
        verbose_output=0;
    case {'h', 60}
        intraday = 1;
        duration = 60;
        dt = 1;
        wps_duration='PT01H';
        verbose_output=0;
    case 'd'
        intraday = 0;
        dt = 24;
        wps_duration='P01D';
        verbose_output=0;
    otherwise
        error('Incorrect time step parameter');
end

% Build CAMS Radiation WPS request
cmd = sprintf('%s/service/wps?Service=WPS&Request=Execute&Identifier=get_cams_radiation&version=1.0.0&DataInputs', server);
cmd = sprintf('%s=latitude=%f;longitude=%f;altitude=%f', cmd, geopoint(:,1), geopoint(:,2), geopoint(:,3));
cmd = sprintf('%s;date_begin=%s', cmd, start_date);
cmd = sprintf('%s;date_end=%s', cmd, end_date);
cmd = sprintf('%s;time_ref=%s', cmd, time);
cmd = sprintf('%s;summarization=%s', cmd, wps_duration);

% encode '@' character in email address
tmp = split(email_address,'@');
username = tmp{1};
domain   = tmp{2};
cmd = sprintf('%s;username=%s%%2540%s', cmd, username, domain);

if verbose_output
    cmd = sprintf('%s;verbose=%d', cmd, verbose_output);
end
cmd = sprintf('%s&RawDataOutput=irradiation', cmd);

% call CAMS Radiation WPS via wget
% note: we use wget -nv option and 2>&1 to get error messages
fic_out = tempname; % output file = temporary file name
cmd_dos = sprintf('wget -O %s -nv "%s" 2>&1', fic_out, cmd);
fprintf('Run command: %s\n', cmd_dos); % trace
[status, result] = system2(cmd_dos);

% Test if wget error
if status ~= 0
   fprintf(1,'ERROR: %s\n', result);
   return
end

% Debug trace
% type(fic_out)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Test if error in output file
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Test if missing file
fp = fopen(fic_out, 'r');
if fp<0
   fprintf(1,'ERROR: cannot create temporary file!\n')
   return
end

% Test if empty file
fseek(fp,0,1); % goto EOF
position = ftell(fp); % get current position in file
if position==0
   fclose(fp);
   delete(fic_out);
   fprintf(1,'ERROR: temporary file is empty!\n');
   return
end

% error if output file is XML
fseek(fp,0,-1); % "rewinds" the file
line=fgets(fp);
if ~isempty(strfind(line, 'xml'))
   fclose(fp);
   type(fic_out) % print error message
   delete(fic_out);
   return
end

fclose(fp);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read output file header
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% read ground elevation on line 14
nb_headerlines = 13;% skip 13 lines
fp = fopen(fic_out,'r');
C = textscan(fp,'%s %n','Delimiter',':','HeaderLines',nb_headerlines);
geopoint(:,3) = C{2}(1);
fclose(fp);

% read version on line 3
nb_headerlines = 2;% skip 2 lines
fp = fopen(fic_out, 'r');
C = textscan(fp,'%s',2,'Delimiter',':','HeaderLines',nb_headerlines);
version = sscanf(cell2mat(C{1}(2)), 'CAMS Radiation Service v%f');% parse column 2
fclose(fp);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read output file in C array
%
% CAMS Radiation v2 columns:
% 1. Observation period (ISO 8601)
% 2. TOA. Irradiation on horizontal plane at the top of atmosphere (Wh/m2)
% 3. Clear sky GHI. Clear sky global irradiation on horizontal plane at ground level (Wh/m2)
% 4. Clear sky BHI. Clear sky beam irradiation on horizontal plane at ground level (Wh/m2)
% 5. Clear sky DHI. Clear sky diffuse irradiation on horizontal plane at ground level (Wh/m2)
% 6. Clear sky BNI. Clear sky beam irradiation on mobile plane following the sun at normal incidence (Wh/m2)
% 7. GHI. Global irradiation on horizontal plane at ground level (Wh/m2)
% 8. BHI. Beam irradiation on horizontal plane at ground level (Wh/m2)
% 9. DHI. Diffuse irradiation on horizontal plane at ground level (Wh/m2)
%10. BNI. Beam irradiation on mobile plane following the sun at normal incidence (Wh/m2)
%11. Reliability. Proportion of reliable data in the summarization (0-1)
%12. sza. Solar zenith angle for the middle of the summarization (deg)
%13. atm. Atmospheric profile code: afglus=U.S. standard afglt=tropical afglms=midlatitude summer afglmw=midlatitude winter afglss=subarctic summer afglsw=subarctic winter
%14. tco3. Total column content of ozone (Dobson unit)
%15. tcwv. Total column content of water vapour (kg/m2)
%16. AOD BC. Partial aerosol optical depth at 550 nm for black carbon
%17. AOD DU. Partial aerosol optical depth at 550 nm for dust
%18. AOD SS. Partial aerosol optical depth at 550 nm for sea salt
%19. AOD OR. Partial aerosol optical depth at 550 nm for organic matter
%20. AOD SU. Partial aerosol optical depth at 550 nm for sulphate
%21. AOD 550. Aerosol optical depth at 550 nm
%22. AOD 1240. Aerosol optical depth at 1240 nm
%23. alpha. Angstroem coefficient for aerosol
%24. Aerosol type. Type of aerosol: -1=no value 5=urban 7=continental clean 8=continental polluted 9=continental average 10=maritime clean 11= maritime polluted 12=maritime tropical 13=antarctic 14=desert
%25. fiso. MODIS-like BRDF parameter fiso
%26. fvol. MODIS-like BRDF parameter fvol
%27. fgeo. MODIS-like BRDF parameter fgeo
%28. albedo. Ground albedo
%29. Cloud optical depth  (value of the nearest acquisition time of the pixel)
%30. Cloud coverage of the pixel (percentage from 0 to 100, value of the nearest acquisition time of the pixel)
%31. Cloud type (value of the nearest acquisition time of the pixel) -1=no value 0=no clouds 5=low-level cloud 6=medium-level cloud 7=high-level cloud 8=thin cloud
%32. GHI no corr. Global irradiation without bias correction on horizontal plane at ground level (Wh/m2)
%33. BHI no corr. Beam irradiation without bias correction on horizontal plane at ground level (Wh/m2)
%34. DHI no corr. Diffuse irradiation without bias correction on horizontal plane at ground level (Wh/m2)
%35. BNI no corr. Beam irradiation without bias correction on mobile plane following the sun at normal incidence (Wh/m2)
%
% Columns 12 to 31 are available if 1-minute UT time serie with -v option.
% Columns 32 to 35 are available if 1-minute UT time serie with -v option if CAMS Radiation version >= 3.
%
% In McClear v3, the column 13 is changed to:
% 13. summer/winter split. 1.0 means summer, 0.0 means winter
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Read columns
fp = fopen(fic_out,'rt');
if (verbose_output && version >= 3) % if 35 columns are available, read only columns 1 to 31
    C = textscan(fp,'%s %f %f %f %f %f %f %f %f %f %f %f %s %f %f %f %f %f %f %f %f %f %f %d %f %f %f %f %f %f %f %*f %*f %*f %*f\n', 'CommentStyle','#', 'Delimiter',';');
elseif (verbose_output) % read columns 1 to 31
    C = textscan(fp,'%s %f %f %f %f %f %f %f %f %f %f %f %s %f %f %f %f %f %f %f %f %f %f %d %f %f %f %f %f %f %f\n', 'CommentStyle','#', 'Delimiter',';');
else % read columns 1 to 11
    C = textscan(fp,'%s %f %f %f %f %f %f %f %f %f %f\n', 'CommentStyle','#', 'Delimiter',';');
end
fclose(fp);

nbval = length(C{1});

delete(fic_out);

%%%%%%%%%%%%%%%%%%%%%%%
% fill HC4 structure
%%%%%%%%%%%%%%%%%%%%%%%

jd0 = ymd_to_jd([ymd_begin(1:3) 0]); % first day at 00:00 (julian day)
jd1 = ymd_to_jd([ymd_end(1:3) 0]); % last day at 00:00 (julian day)
vjd_h0 = jd0:jd1; % list of julian days
ymd = jd_to_ymd(vjd_h0); % list of ymd days
ymd(:, 4) = []; % remove last column

HC4.name = ''; % obsolete
HC4.geopoint = geopoint;
HC4.time = time;
HC4.command = cmd; % command line called

if (intraday == 1)
    t = [dt:dt:24]; % list of times for 1 day (end of period)
    nt = length(t); % #data per day
else
    switch(duration)
        case 'd'
            t = 24; % list of times for 1 day (end of period)
            nt = length(t); % #data per day

        otherwise
            error('Incorrect time step parameter');
    end
end

% decode Observation period = Start date/End date
datenums=zeros(nbval, 1); % list of ymdhms times
for i=1:nbval
    D=textscan(C{1}{i}, '%*s %s', 'delimiter', '/');
    datenums(i)=datenum(D{1}, 'yyyy-mm-ddTHH:MM:SS');
end
ymdhms=datevec(datenums); % list of ymdhms times

C2 = C{1,2}; % TOA
C3 = C{1,3}; % CLS_GHI
C4 = C{1,4}; % CLS_BHI
C5 = C{1,5}; % CLS_DHI
C6 = C{1,6}; % CLS_BNI
C7 = C{1,7}; % GHI
C8 = C{1,8}; % BHI
C9 = C{1,9}; % DHI
C10= C{1,10};% BNI

if (verbose_output) % if columns 12 to 31 are available
    C11 = C{1,11}; % reliability
    C12 = C{1,12}; % sza
    C13 = C{1,13}; % atm in CAMS Radiation v2, summer/winter split in v3
    C14 = C{1,14}; % tco3
    C15 = C{1,15}; % tcwv
    C16 = C{1,16}; % AOD BC
    C17 = C{1,17}; % AOD DU
    C18 = C{1,18}; % AOD SS
    C19 = C{1,19}; % AOD OR
    C20 = C{1,20}; % AOD SU
    C21 = C{1,21}; % AOD 550
    C22 = C{1,22}; % AOD 1240
    C23 = C{1,23}; % alpha
    C24 = C{1,24}; % Aerosol type
    C25 = C{1,25}; % fiso
    C26 = C{1,26}; % fvol
    C27 = C{1,27}; % fgeo
    C28 = C{1,28}; % albedo
    C29 = C{1,29}; % cloud optical depth
    C30 = C{1,30}; % cloud coverage
    C31 = C{1,31}; % cloud type
end

HC4.ymd = ymd; % list of ymd days
HC4.t = t; % list of times for 1 day (end of period)
HC4.nt = nt; % #data per day
HC4.ymdhms = reshape(ymdhms,[nt*length(vjd_h0) 6]); % list of ymdhms times

HC4.TOA     = reshape(C2, [nt length(vjd_h0)]);
HC4.CLS_GHI = reshape(C3, [nt length(vjd_h0)]);
HC4.CLS_BHI = reshape(C4, [nt length(vjd_h0)]);
HC4.CLS_DHI = reshape(C5, [nt length(vjd_h0)]);
HC4.CLS_BNI = reshape(C6, [nt length(vjd_h0)]);
HC4.GHI     = reshape(C7,[nt length(vjd_h0)]);
HC4.BHI     = reshape(C8,[nt length(vjd_h0)]);
HC4.DHI     = reshape(C9,[nt length(vjd_h0)]);
HC4.BNI     = reshape(C10,[nt length(vjd_h0)]);

if (verbose_output) % if columns 12 to 31 are available
    HC4.reliability = reshape(C11, [nt length(vjd_h0)]);
    
    HC4.sza   = reshape(C12, [nt length(vjd_h0)]);
    HC4.sza(HC4.sza == 0) = nan;
    
    if version < 3 % if version v2.x
        HC4.atm = reshape(C13, [nt length(vjd_h0)]);
    else
        C13 = str2num(cell2mat(C13));
        HC4.summer_winter_split = reshape(C13, [nt length(vjd_h0)]);
    end
    
    HC4.tco3 = reshape(C14, [nt length(vjd_h0)]);
    HC4.tcwv = reshape(C15, [nt length(vjd_h0)]);
    HC4.aod_bc = reshape(C16, [nt length(vjd_h0)]);
    HC4.aod_du = reshape(C17, [nt length(vjd_h0)]);
    HC4.aod_ss = reshape(C18, [nt length(vjd_h0)]);
    HC4.aod_or = reshape(C19, [nt length(vjd_h0)]);
    HC4.aod_su = reshape(C20, [nt length(vjd_h0)]);
    HC4.aod_550 = reshape(C21, [nt length(vjd_h0)]);
    HC4.aod_1240 = reshape(C22, [nt length(vjd_h0)]);
    HC4.alpha = reshape(C23, [nt length(vjd_h0)]);
    HC4.aerosol_type = reshape(C24, [nt length(vjd_h0)]);
    HC4.fiso = reshape(C25, [nt length(vjd_h0)]);
    HC4.fvol = reshape(C26, [nt length(vjd_h0)]);
    HC4.fgeo = reshape(C27, [nt length(vjd_h0)]);
    HC4.albedo = reshape(C28, [nt length(vjd_h0)]);
    HC4.cloud_optical_depth = reshape(C29, [nt length(vjd_h0)]);
    HC4.cloud_coverage = reshape(C30, [nt length(vjd_h0)]);
    HC4.cloud_type = reshape(C31, [nt length(vjd_h0)]);
end
