function MC = mcclear_wget(email_address, geopoint, start_date, end_date, duration, verbose_output, time, server)
% MC = mcclear_wget(email_address, geopoint, start_date, end_date, duration, verbose_output, time, server)
%
% Description:
% Use McClear WPS to retrieve clear-sky IRRADIATION data (Wh/m2) from SoDa distant server
%
% Inputs: 
% email_address  : email, 'myname@mydomain.com' (must be subscribed to www.soda-pro.com)
% geopoint       : [lat lon] or [lat lon h] in deg/deg/m
% start_date     : start date, format 'YYYY-MM-DD'
% end_date       : end date, format 'YYYY-MM-DD'
% duration       : time step: month 'm', day 'd', hour 'h', 15 or 1 minutes
% verbose_output : boolean to retrieve other information such as aerosols; only applicable to 'UT' and 1-min
% time           : time reference 'UT' or 'TST'
% server         : SoDa server. Default is 'http://www.soda-is.com'.
%
% Outputs:
% MC : structure containing location, period, IRRADIATION (in Wh/m2), etc.
%
% Example:
% MC = mcclear_wget('myname@mydomain.com', [45.0 25.0], '2008-08-01', '2008-08-03', 15)

MC = [];

% default values
if (nargin < 6)
    verbose_output = 0;
end
if (nargin < 7)
    time = 'UT';
end
if (nargin < 8)
    server = 'http://www.soda-is.com';
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Call McClear WPS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Convert dates to format [yyyy doy]
tmp = split(start_date,'-');
ymd_begin = [str2double(tmp{1}) str2double(tmp{2}) str2double(tmp{3})];
tmp = split(end_date,'-');
ymd_end = [str2double(tmp{1}) str2double(tmp{2}) str2double(tmp{3})];

% if elevation = -999 or is missing, SRTM elevation will be used
if (size(geopoint,2)==2) || (size(geopoint,2)==3 && geopoint(3) < -900)
    geopoint(:,3) = -999;
end

% McClear WPS expects time reference = UT or TST
switch(time)
    case 'TST'
        ; % nothing to do
    case {'TU', 'UT'}
        time = 'UT';
    otherwise
        error('Incorrect time reference');
end

% parse duration and check verbose_output
switch(duration)
    case 1 % min
        intraday = 1;
        dt = duration/60; % duration in hours
        wps_duration='PT01M';
        if ~strcmp(time, 'UT')
            verbose_output=0; % -v option is supported if 1-minute UT time series only
        end
    case 15 % min
        intraday = 1;
        dt = duration/60;
        wps_duration='PT15M';
        verbose_output=0;
    case {'h', 60}
        intraday = 1;
        duration = 60;
        dt = 1;
        wps_duration='PT01H';
        verbose_output=0;
    case 'd'
        intraday = 0;
        dt = 24;
        wps_duration='P01D';
        verbose_output=0;
    otherwise
        error('Incorrect time step parameter');
end

% Build McClear WPS request
cmd = sprintf('%s/service/wps?Service=WPS&Request=Execute&Identifier=get_mcclear&version=1.0.0&DataInputs', server);
cmd = sprintf('%s=latitude=%f;longitude=%f;altitude=%f', cmd, geopoint(:,1), geopoint(:,2), geopoint(:,3));
cmd = sprintf('%s;date_begin=%s', cmd, start_date);
cmd = sprintf('%s;date_end=%s', cmd, end_date);
cmd = sprintf('%s;time_ref=%s', cmd, time);
cmd = sprintf('%s;summarization=%s', cmd, wps_duration);

% encode '@' character in email address
tmp = split(email_address,'@');
username = tmp{1};
domain   = tmp{2};
cmd = sprintf('%s;username=%s%%2540%s', cmd, username, domain);

if verbose_output
    cmd = sprintf('%s;verbose=%d', cmd, verbose_output);
end
cmd = sprintf('%s&RawDataOutput=irradiation', cmd);

% call McClear WPS via wget
% note: we use wget -nv option and 2>&1 to get error messages
fic_out = tempname; % output file = temporary file name
cmd_dos = sprintf('wget -O %s -nv "%s" 2>&1', fic_out, cmd);
fprintf('Run command: %s\n', cmd_dos); % trace
[status, result] = system2(cmd_dos);

% Test if wget error
if status ~= 0
   fprintf(1,'ERROR: %s\n', result);
   return
end

% Debug trace
% type(fic_out)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Test if error in output file
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Test if missing file
fp = fopen(fic_out, 'r');
if fp<0
   fprintf(1,'ERROR: cannot create temporary file!\n')
   return
end

% Test if empty file
fseek(fp,0,1); % goto EOF
position = ftell(fp); % get current position in file
if position==0
   fclose(fp);
   delete(fic_out);
   fprintf(1,'ERROR: temporary file is empty!\n');
   return
end

% error if output file is XML
fseek(fp,0,-1); % "rewinds" the file
line=fgets(fp);
if ~isempty(strfind(line, 'xml'))
   fclose(fp);
   type(fic_out) % print error message
   delete(fic_out);
   return
end

fclose(fp);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read output file header
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% read ground elevation on line 14
nb_headerlines = 13;% skip 13 lines
fp = fopen(fic_out,'r');
C = textscan(fp,'%s %n','Delimiter',':','HeaderLines',nb_headerlines);
geopoint(:,3) = C{2}(1);
fclose(fp);

% read version on line 3
nb_headerlines = 2;% skip 2 lines
fp = fopen(fic_out, 'r');
C = textscan(fp,'%s',2,'Delimiter',':','HeaderLines',nb_headerlines);
version = sscanf(cell2mat(C{1}(2)), 'CAMS McClear v%f'); % parse column 2
fclose(fp);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read output file in C array
%
% McClear v2 columns:
% 1. Observation period (ISO 8601)
% 2. TOA. Irradiation on horizontal plane at the top of atmosphere (Wh/m2)
% 3. Clear sky GHI. Clear sky global irradiation on horizontal plane at ground level (Wh/m2)
% 4. Clear sky BHI. Clear sky beam irradiation on horizontal plane at ground level (Wh/m2)
% 5. Clear sky DHI. Clear sky diffuse irradiation on horizontal plane at ground level (Wh/m2)
% 6. Clear sky BNI. Clear sky beam irradiation on mobile plane following the sun at normal incidence (Wh/m2)
% 7. sza. Solar zenith angle for the middle of the summarization (deg)
% 8. atm. Atmospheric profile code: afglus=U.S. standard afglt=tropical afglms=midlatitude summer afglmw=midlatitude winter afglss=subarctic summer afglsw=subarctic winter
% 9. tco3. Total column content of ozone (Dobson unit)
%10. tcwv. Total column content of water vapour  (kg/m2)
%11. AOD BC. Partial aerosol optical depth at 550 nm for black carbon
%12. AOD DU. Partial aerosol optical depth at 550 nm for dust
%13. AOD SS. Partial aerosol optical depth at 550 nm for sea salt
%14. AOD OR. Partial aerosol optical depth at 550 nm for organic matter
%15. AOD SU. Partial aerosol optical depth at 550 nm for sulphate
%16. AOD 550. Aerosol optical depth at 550 nm
%17. AOD 1240. Aerosol optical depth at 1240 nm
%18. alpha. Angstroem coefficient for aerosol
%19. Aerosol type. Type of aerosol: -1=no value 5=urban 7=continental clean 8=continental polluted 9=continental average 10=maritime clean 11= maritime polluted 12=maritime tropical 13=antarctic 14=desert
%20. fiso. MODIS-like BRDF parameter fiso
%21. fvol. MODIS-like BRDF parameter fvol
%22. fgeo. MODIS-like BRDF parameter fgeo
%23. albedo. Ground albedo
%
% Columns 7 to 23 are available if 1-minute UT time serie with -v option.
%
% In McClear v3, the column 8 is changed to:
% 8. summer/winter split. 1.0 means summer, 0.0 means winter
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Read columns
fp = fopen(fic_out,'rt');
if (verbose_output) % if columns 7 to 23 are available
    C = textscan(fp,'%s %f %f %f %f %f %f %s %f %f %f %f %f %f %f %f %f %f %d %f %f %f %f\n', 'CommentStyle','#', 'Delimiter',';');
else
    C = textscan(fp,'%s %f %f %f %f %f\n', 'CommentStyle','#', 'Delimiter',';');
end
fclose(fp);

nbval = length(C{1});

delete(fic_out);

%%%%%%%%%%%%%%%%%%%%%%%
% fill MC structure
%%%%%%%%%%%%%%%%%%%%%%%

jd0 = ymd_to_jd([ymd_begin(1:3) 0]); % first day at 00:00 (julian day)
jd1 = ymd_to_jd([ymd_end(1:3) 0]); % last day at 00:00 (julian day)
vjd_h0 = jd0:jd1; % list of julian days
ymd = jd_to_ymd(vjd_h0); % list of ymd days
ymd(:, 4) = []; % remove last column

MC.name = 'Broadband McClear values';
MC.geopoint = geopoint;
MC.time = time;
MC.command = cmd; % command line called

if (intraday == 1)
    t = [dt:dt:24]; % list of times for 1 day (end of period)
    nt = length(t); % #data per day
else
    switch(duration)
        case 'd'
            t = 24; % list of times for 1 day (end of period)
            nt = length(t); % #data per day

        otherwise
            error('Incorrect time step parameter');
    end
end

% decode Observation period = Start date/End date
datenums=zeros(nbval, 1); % list of ymdhms times
for i=1:nbval
    D=textscan(C{1}{i}, '%*s %s', 'delimiter', '/');
    datenums(i)=datenum(D{1}, 'yyyy-mm-ddTHH:MM:SS');
end
ymdhms=datevec(datenums); % list of ymdhms times

C2 = C{1,2}; % TOA
C3 = C{1,3}; % GHI
C4 = C{1,4}; % BHI
C5 = C{1,5}; % DHI
C6 = C{1,6}; % BNI

if (verbose_output) % if columns 7 to 23 are available
    C7 = C{1,7}; % sza
    C8 = C{1,8}; % atm in McClear v2, summer/winter split in v3
    C9 = C{1,9}; % tco3
    C10 = C{1,10}; % tcwv
    C11 = C{1,11}; % AOD BC
    C12 = C{1,12}; % AOD DU
    C13 = C{1,13}; % AOD SS
    C14 = C{1,14}; % AOD OR
    C15 = C{1,15}; % AOD SU
    C16 = C{1,16}; % AOD 550
    C17 = C{1,17}; % AOD 1240
    C18 = C{1,18}; % alpha
    C19 = C{1,19}; % Aerosol type
    C20 = C{1,20}; % fiso
    C21 = C{1,21}; % fvol
    C22 = C{1,22}; % fgeo
    C23 = C{1,23}; % albedo
end

MC.ymd_begin = ymd(1,:);
MC.ymd_end = ymd(end,:);
MC.nbday = length(ymd(:,1));
MC.ymdhms = reshape(ymdhms,[nt*length(vjd_h0) 6]); % list of ymdhms times
MC.ymd = ymd; % list of ymd days
MC.t = t; % list of times for 1 day (end of period)
MC.dt = dt;
MC.nt = nt; % #data per day
MC.y = ymd_to_y(ymd);
MC.jd = ymd_to_jd(ymd);
MC.time = time;

MC.TOA = reshape(C2, [nt length(vjd_h0)]);
MC.GHI = reshape(C3, [nt length(vjd_h0)]);
MC.BHI = reshape(C4, [nt length(vjd_h0)]);
MC.DHI = reshape(C5, [nt length(vjd_h0)]);
MC.BNI = reshape(C6, [nt length(vjd_h0)]);

if (verbose_output) % if columns 7 to 23 are available
    MC.sza   = reshape(C7, [nt length(vjd_h0)]);
    MC.sza(MC.sza == 0) = nan;
    
    if version < 3 % if version v2.x
        MC.atm = reshape(C8, [nt length(vjd_h0)]);
    else
        C8 = str2num(cell2mat(C8));
        MC.summer_winter_split = reshape(C8, [nt length(vjd_h0)]);
    end
    
    MC.tco3 = reshape(C9, [nt length(vjd_h0)]);
    MC.tcwv = reshape(C10, [nt length(vjd_h0)]);
    MC.aod_bc = reshape(C11, [nt length(vjd_h0)]);
    MC.aod_du = reshape(C12, [nt length(vjd_h0)]);
    MC.aod_ss = reshape(C13, [nt length(vjd_h0)]);
    MC.aod_or = reshape(C14, [nt length(vjd_h0)]);
    MC.aod_su = reshape(C15, [nt length(vjd_h0)]);
    MC.aod_550 = reshape(C16, [nt length(vjd_h0)]);
    MC.aod_1240 = reshape(C17, [nt length(vjd_h0)]);
    MC.alpha = reshape(C18, [nt length(vjd_h0)]);
    MC.aerosol_type = reshape(C19, [nt length(vjd_h0)]);
    MC.fiso = reshape(C20, [nt length(vjd_h0)]);
    MC.fvol = reshape(C21, [nt length(vjd_h0)]);
    MC.fgeo = reshape(C22, [nt length(vjd_h0)]);
    MC.albedo = reshape(C23, [nt length(vjd_h0)]);
end

