function [lon,lat] = MSG_loc_dir(qx, qy, h, lon_sat)
%MSG_loc_dir(qx, qy, h, lon_sat)
% Convert Meteosat pixel in [0.5, 3712.5] x [0.5, 3712.5] -> (lat,lon) in degrees.
%
% This is the formula used by Helioclim 3. There is a 0.5 pixel shift compared to:
%   [1] LRIT/HRIT Global Specification, Issue 2.6, August 1999.
%
% qx: x in [0.5, 3712.5]
% qy: y in [0.5, 3712.5]
% h: altitude in meters (default is 0)
% lon_sat: longitude of geostationary satellite in degrees (default is 0)

if (nargin < 3)
    h = 0;
end
if (nargin < 4)
    lon_sat = 0;
end

r_pol = 6356.5838; % perfect ellipsoid polar radius in km [1]
r_eq = 6378.1690; % perfect ellipsoid equator radius in km [1]
hsat = 42164; % satellite distance from center of earth in km from [1]
npy = 3712; 
npx = 3712;
fac = 781648343.404;
deltaX = -2^16/fac;
deltaY = -2^16/fac;
k1 = (r_eq/r_pol)^2;
k2 = 1737121856;

px = npx-qx+1-0.5;
py = npy-qy+1-0.5;

x = (px-npx/2)*deltaX;
y = (py-npy/2)*deltaY;

cosx = cos(x);
cosy = cos(y);
siny = sin(y);
siny2 = siny.^2;
cosy2 = cosy.^2;

sd = sqrt((hsat*cosx.*cosy).^2 - (cosy2 + k1*siny2)*k2);
sn = (hsat*cosx.*cosy - sd)./(cosy2 + k1*siny2);
s1 = hsat - sn.*cosx.*cosy;
s2 = sn.*sin(x).*cosy;
s3 = -sn.*siny;
sxy = sqrt(s1.^2 + s2.^2);

lon = atan(s2./s1)*180/pi + lon_sat;
lat = atan(k1*s3./sxy)*180/pi;

