function [Mm,ym,s] = agglo_d2m(Md, ymd, G0d)
% [Mm,ym,s] = agglo_d2m(Md, ymd, G0d)
%
% Sum daily irradiations to compute monthly irradiation. Approximate missing days with clear sky irradiations.
%
% Parameters:
% Md:   daily irradiations
% ymd:  list of year/month/day days
% G0d:  daily clear sky irradiations
%
% Output:
% Mm: monthly irradiation, sum of daily irradiations
% ym: list of year/month months
% s.rNB: ratio (nb VALID days)/(nb days) for each month
% s.rG0d: ratio (sum of VALID daily clear sky irradiations)/(sum of daily clear sky irradiations) for each month

nbd = size(ymd,1); % #days in ymd
nc = size(Md,2); % #columns in Md

if (size(Md,1) ~= nbd)
    error('M doit avoir autant de ligne que ymd');
end
if (nargin < 3)
    G0d = ones(size(Md));
end

if (size(G0d,1) ~= nbd)
    error('G0d doit avoir autant de ligne que ymd');
end
if (size(G0d,2) == 1)
    G0d = repmat(G0d,[1 nc]);
end

okMd = ~isnan(Md); % VALID values in Md
Md(~okMd) = 0; % replace NaNs by 0 in Md

yu = unique(ymd(:,1)); % get list of years in ymd
nyu = length(yu); % #years in ymd

% allocate result arrays
ym = zeros(12*nyu,2);
Mm = nan(12*nyu,nc);
if (nargout >= 2)
    s.rNB = zeros(12*nyu,nc);
    s.rG0d = zeros(12*nyu,nc);
end

for kyu = 1:nyu % for each year in ymd
    
    idx_kyu = find(ymd(:,1)==yu(kyu)); % find days belonging to year kyu
    ymd_kyu = ymd(idx_kyu,:); % copy days belonging to year kyu
    vmm = [1:12]'; % list of months
    ym((kyu-1)*12 + vmm,:) = [yu(kyu)*ones(12,1) vmm]; % fill year and months in ym for year kyu
    
    m_kyu = repmat(ymd_kyu(:,2),1,nc); % month for each day of year kyu
    Md_kyu = Md(idx_kyu,:); % irradiation values for year kyu
    G0d_kyu = G0d(idx_kyu,:); % clear sky values for year kyu
    okMd_kyu = okMd(idx_kyu,:); % VALID values for year kyu
    
    for km = 1:length(vmm) % for each month
        nbm_nominal = nbday(yu(kyu),vmm(km)); % #days in month km
        K_kyu_km_nominal = (m_kyu == vmm(km)); % list of days in month km
        K_kyu_km_effectif = K_kyu_km_nominal&okMd_kyu; % list of VALID days in month km
        nbm_effectif = sum(K_kyu_km_effectif,1); % nb VALID days in month km
        
        sG0d_nominal = sum(G0d_kyu.*K_kyu_km_nominal,1); % sum of daily clear sky irradiations for month km
        sG0d_effectif = sum(G0d_kyu.*K_kyu_km_effectif,1); % sum of VALID daily clear sky irradiations for month km
        
        % ratio (sum of VALID daily clear sky irradiations)/(sum of daily clear sky irradiations) for month km
        r = ones(size(sG0d_effectif));
        r(sG0d_nominal~=0) = sG0d_effectif(sG0d_nominal~=0)./sG0d_nominal(sG0d_nominal~=0);
        s.rG0d((kyu-1)*12 + vmm(km),:) = r;
        s.rG0d((kyu-1)*12 + vmm(km),nbm_effectif==0) = 0;
        
        % ratio (nb VALID days)/(nb days) for month km
        s.rNB((kyu-1)*12 + vmm(km),:) = nbm_effectif/nbm_nominal;
        
        % Sum daily irradiations to compute monthly irradiation. Approximate missing days with clear sky irradiations.
        Mm((kyu-1)*12 + vmm(km),:) = sum(Md_kyu.*K_kyu_km_effectif,1) ./ r;
        Mm((kyu-1)*12 + vmm(km),nbm_effectif==0) = nan;
    end
    
end

