function w = blackman_H4T(varargin)

[w,msg] = gcw(varargin{:});

function [w,msg] = gcw(varargin)
%GENCOSWIN   Returns one of the generalized cosine windows.
%   GENCOSWIN returns the generalized cosine window specified by the 
%   first string argument. Its inputs can be
%     Window name    - a string, any of 'hamming', 'hann', 'blackman'.
%     N              - length of the window desired.
%     Sampling flag  - optional string, one of 'symmetric', 'periodic'. 

%   Copyright (c) 1988-1999 The MathWorks, Inc. All Rights Reserved. 
%   $Revision: 4480 $  $Date: 2017-01-05 15:35:42 +0100 (Thu, 05 Jan 2017) $ 

% Parse the inputs
n = varargin{1};
msg = '';

% Select the sampling option
if nargin == 1, % no sampling flag specified, use default. 
   sflag = 'symmetric';
else
   sflag = lower(varargin{2});
end

% Allow partial strings for sampling options
allsflags = {'symmetric','periodic'};
sflagindex = strmatch(sflag, allsflags);
if length(sflagindex)~=1         % catch 0 or 2 matches
   msg = 'Sampling flag must be either ''symmetric'' or ''periodic''.';
   return;
else	
   sflag = allsflags{sflagindex};
end

% Evaluate the window
switch sflag
case 'periodic'
   w = sw(n+1);
   w(end) = [];
case 'symmetric'
   w = sw(n);
end

%---------------------------------------------------------------------
function w = sw(n)
%SYM_WINDOW   Symmetric generalized cosine window.
%   SYM_WINDOW Returns an exactly symmetric N point generalized cosine 
%   window by evaluating the first half and then flipping the same samples
%   over the other half.

if ~rem(n,2)
   % Even length window
   half = n/2;
   w = cw(half,n);
   w = [w; w(end:-1:1)];
else
   % Odd length window
   half = (n+1)/2;
   w = cw(half,n);
   w = [w; w(end-1:-1:1)];
end

%---------------------------------------------------------------------
function w = cw(m,n)

w = (0.40217 - 0.4970*cos(2*pi*(0:m-1)/(n-1)) + 0.09892*cos(4*pi*(0:m-1)/(n-1)) - ...
   0.00188*cos(6*pi*(0:m-1)/(n-1)))';

% [EOF] gencoswin.m