function SG = calc_sg(gp, ymd, tu, vtl0, TZ, mod)
%SG = calc_sg(gp, ymd, tu, vtl0, TZ, mod)
% Call solar_angles to get solar angles in radians at the MIDDLE of the previous
% interval and extraterrestrial IRRADIATIONS in Wh/m2.
% gp : [lat lon elev]
% ymd : [Nx3] ymd table for N days
% tu : table of TU times for 1 day
% vtl0 : Linke turbidity
%  - empty [] => no cls computed
%  - scalar > 0 same TL for every day 
%  - scalar = -1 => use of get_TL5 at the given location
%  - 12x1 or 1x12 => use of daily value of TL from linear interp of the monthly TL.
% TZ : time zone (optional)
% mod : 1 for modified ESRA (default), 0 for plain ESRA, like HelioClim-3
%
% NB: solar angles are given at the middle of the previous time interval to
% be in adequation with Matlab function providing radiation values (HC3,
% CAMS...). For instances, at a hourly summarization, solar angles given at 
% 08:00 correspond to the angles at the instant 07:30. Online Web
% interface on SoDa delivers instantaneous values at the EXACT time step
% requested by the user (here: 08:00), which could explain discrepancies 
% observed using Matlab / online services. 

% default values
if (nargin < 4)
    vtl0 = [];
end
if (nargin < 5)
    TZ = 0;
end
if (nargin < 6)
    mod = 1;
end
if (isempty(TZ))
    TZ = 0;
end

% if elevation = -999 or is missing, get SRTM or GTPOP30 altitude
if (size(gp,2)==2) || (size(gp,2)==3 && gp(3) < -900)
    gp(:,3) = select_altitude(gp(1), gp(2));
    fprintf('get SRTM or GTOPO30 altitude: %f\n', gp(3)); % trace
end

nbj = size(ymd,1); % number of days
jd = ymd_to_jd(ymd); % list of julian days
nt = length(tu); % number of intervals per day
dtu = tu(2)-tu(1); % interval length

Tst = zeros(nt, nbj);
AlphaS = zeros(nt, nbj);
GammaS = zeros(nt, nbj);
CosThetaS = zeros(nt, nbj);

% if vtl0 is provided, create a function to interpolate the Linke trouble
calc_cls = 1;
if (isempty(vtl0))
    calc_cls = 0;
else
    if (length(vtl0) == 1)
        if (vtl0 < 0)
            vtl0 = get_TL5(gp(1),gp(2),[1:12]);
        else
            vtl0 = repmat(vtl0,[1 12]);
        end
    end
    vtl0 = vtl0(:)';
    vtl = [vtl0(end) vtl0 vtl0(1)];
    vmm = [0:13];
    interp_tl = inline('interp1(vmm+0.5, vtl, d(:,2) + (d(:,3)-1)./nbday(d(:,1),d(:,2)), ''lin'', nan)', 'vmm', 'vtl', 'd');
end

GHItoa = zeros(nt, nbj);
if (calc_cls)
    BHIcls = zeros(nt, nbj);
    DHIcls = zeros(nt, nbj);
    GHIcls = zeros(nt, nbj);
end

Ecorr = zeros(nbj,1);
omega_ss = zeros(nbj,1);

for kj = 1:nbj % for each day
    
    jd_kj = ymd_to_jd([ymd(kj,:) 0]);
    sd = solar_day(gp(1),gp(2),ymd(kj,3),ymd(kj,2),ymd(kj,1));
    omega_ss(kj) = sd.omega_s(2);
    delta(kj) = sd.declination;
    
    tst = tu(:) + TZ - sd.dt_tst2tu;
    idx_dbefore = find(tst <= 0);
    idx_d = find((tst > 0)&(tst <= 24));
    idx_dafter = find(tst > 24);
 
    ymd_ = jd_to_ymd(jd_kj);
    idx_ = idx_d;
    tst_ = tst(idx_);
    
    if (~isempty(idx_))        
        [omega, alphas, gammas, cos_thetaS] = solar_angles(gp(1),gp(2),...
            ymd_(3),ymd_(2),ymd_(1),tst_-dtu/2,0,0);
        Tst(idx_,kj) = tst_;
        GammaS(idx_,kj) = gammas(:);
        alphas = convertAz(alphas, 'e2i',gp(1));
        alphas = toRange(alphas,1);
        AlphaS(idx_,kj) = alphas(:);
        CosThetaS(idx_,kj) = cos_thetaS(:);        
        if (calc_cls)
            tl_kj = interp_tl(vmm, vtl, ymd(kj,:));
            [Gcls_kj, Bcls_kj, Dcls_kj] = calc_CLSdt(gp(1),gp(2),...
                ymd_(3),ymd_(2),ymd_(1), gp(3), ...
                tl_kj, [tst_(1)-dtu ; tst_]);
            GHIcls(idx_,kj) = Gcls_kj(2:end);
            BHIcls(idx_,kj) = Bcls_kj(2:end);
            DHIcls(idx_,kj) = Dcls_kj(2:end);
        end        
        G0_kj = calc_H0dt(gp(1),gp(2),ymd_(3),ymd_(2),ymd_(1),[tst_(1)-dtu ; tst_]);
        GHItoa(idx_,kj) = G0_kj(2:end);        
    end
 
    ymd_ = jd_to_ymd(jd_kj+1);
    idx_ = idx_dbefore;
    tst_ = tst(idx_)+24;
    
    if (~isempty(idx_))        
        [omega, alphas, gammas, cos_thetaS] = solar_angles(gp(1),gp(2),...
            ymd_(3),ymd_(2),ymd_(1),tst_-dtu/2,0,0);
        Tst(idx_,kj) = tst_;
        GammaS(idx_,kj) = gammas(:);
        alphas = convertAz(alphas, 'e2i',gp(1));
        alphas = toRange(alphas,1);
        AlphaS(idx_,kj) = alphas(:);
        CosThetaS(idx_,kj) = cos_thetaS(:);        
        if (calc_cls)
            tl_kj = interp_tl(vmm, vtl, ymd(kj,:));
            [Gcls_kj, Bcls_kj, Dcls_kj] = calc_CLSdt(gp(1),gp(2),...
                ymd_(3),ymd_(2),ymd_(1), gp(3), ...
                tl_kj, [tst_(1)-dtu ; tst_]);
            GHIcls(idx_,kj) = Gcls_kj(2:end);
            BHIcls(idx_,kj) = Bcls_kj(2:end);
            DHIcls(idx_,kj) = Dcls_kj(2:end);
        end        
        G0_kj = calc_H0dt(gp(1),gp(2),ymd_(3),ymd_(2),ymd_(1),[tst_(1)-dtu ; tst_]);
        GHItoa(idx_,kj) = G0_kj(2:end);        
    end
 
    ymd_ = jd_to_ymd(jd_kj+1);
    idx_ = idx_dafter;
    tst_ = tst(idx_)-24;
    
    if (~isempty(idx_))        
        [omega, alphas, gammas, cos_thetaS] = solar_angles(gp(1),gp(2),...
            ymd_(3),ymd_(2),ymd_(1),tst_- dtu/2,0,0);
        Tst(idx_,kj) = tst_;
        GammaS(idx_,kj) = gammas(:);
        alphas = convertAz(alphas, 'e2i',gp(1));
        alphas = toRange(alphas,1);
        AlphaS(idx_,kj) = alphas(:);
        CosThetaS(idx_,kj) = cos_thetaS(:);        
        if (calc_cls)
            tl_kj = interp_tl(vmm, vtl, ymd(kj,:));
            [Gcls_kj, Bcls_kj, Dcls_kj] = calc_CLSdt(gp(1),gp(2),...
                ymd_(3),ymd_(2),ymd_(1), gp(3), ...
                tl_kj, [tst_(1)-dtu ; tst_], mod);
            GHIcls(idx_,kj) = Gcls_kj(2:end);
            BHIcls(idx_,kj) = Bcls_kj(2:end);
            DHIcls(idx_,kj) = Dcls_kj(2:end);
        end        
        G0_kj = calc_H0dt(gp(1),gp(2),ymd_(3),ymd_(2),ymd_(1),[tst_(1)-dtu ; tst_]);
        GHItoa(idx_,kj) = G0_kj(2:end);        
    end

    Ecorr(kj) = sd.eccentricity;

end

GammaS0 = GammaS;

Omega_ss = repmat(omega_ss(:)',[nt 1]);

[r0d,r0d_u0d] = calc_r0d(Tst(:)-dtu/2,Omega_ss(:));
R0d = reshape(r0d,size(Tst));
R0d = R0d./repmat(sum(R0d,1),[nt 1]);
R0d_u0d = reshape(r0d_u0d,size(Tst));
R0d_u0d = R0d_u0d./repmat(sum(R0d_u0d,1),[nt 1]);

I0 = 1367.0;
SG.name = 'Solar Geometry';
SG.comment = 'NB: solar angles are given at the middle of the previous time interval';
SG.geopoint = gp;
SG.ymd_begin = ymd(1,1:3);
SG.ymd_end = ymd(end,1:3);
SG.nbday = nbj;
SG.ymd = ymd;
SG.y = ymd_to_y(ymd);
SG.jd = ymd_to_jd(ymd);
SG.t = tu;
SG.dt = tu(2)-tu(1);
SG.nt = nt;
SG.type_t = 'TU';
SG.nbD = 5;
SG.I0Ecorrdt = repmat(I0*Ecorr'*SG.dt(:),nt,1); % correction CT 01/2018
SG.epsilon = repmat(Ecorr',nt,1); % ajout CT 01/2018
SG.omega_ss = omega_ss;
SG.delta = delta;
SG.nomD = [];
SG.GammaS0 = GammaS0;
SG.GammaS = GammaS;
SG.CosThetaS = CosThetaS;
SG.AlphaS = AlphaS;
SG.GHItoa = GHItoa;
SG.R0d = R0d;
SG.R0d_u0d = R0d_u0d;
SG.Tst = Tst;

if (calc_cls)
    SG.GHIcls = GHIcls;
    SG.BHIcls = BHIcls;
    SG.DHIcls =DHIcls;
    SG.nomD = {'GammaS0','GammaS','AlphaS','GHItoa','R0d','R0d_u0d','GHIcls','BHIcls','DHIcls'};
else
    SG.nomD = {'GammaS0','GammaS','AlphaS','GHItoa','R0d','R0d_u0d'};
end
SG.nbD = length(SG.nomD); % number of data arrays
