function h = ftm2ri(varargin)

[H, otfSize, outSize] = ParseInputs(varargin{1:min(nargin,2)});
otf = fftshift(H);

if ~all(otf(:)==0),
   
   psf = ifftn(otf);
%    psf = ifftn(otf,'symmetric');

   % Estimate the rough number of operations involved in the
   % computation of the IFFT
   nElem = prod(otfSize);
   nOps  = 0;
   for k=1:ndims(otf)
	   nffts = nElem/otfSize(k);
	   nOps  = nOps + otfSize(k)*log2(otfSize(k))*nffts;
   end

   % Discard the imaginary part of the psf if it's within roundoff error.
   if max(abs(imag(psf(:))))/max(abs(psf(:))) <= nOps*eps
	   psf = real(psf);
   end

   % Circularly shift psf so that (1,1) element is moved to the
   % appropriate center position.
   psf    = circshift(psf,floor(outSize/2));

   % Crop output array.
   idx = cell(1,length(outSize));
   for k = 1:length(outSize)
	   idx{k} = 1:outSize(k);
   end
   psf = psf(idx{:});

else
	psf = zeros(outSize);
end

if (nargin <= 2)
	h = psf;
else

end




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%555
%%% Parse inputs
%%%

function [otf, otfSize, outSize] = ParseInputs(varargin)

% error(nargchk(1,2,nargin,'struct'));

switch nargin
case 1       % OTF2PSF(OTF) 
  otf = varargin{1};   
case 2       % OTF2PSF(OTF,OUTSIZE) 
  otf = varargin{1}; 
  outSize = varargin{2};
  if (length(outSize) == 1)
	  if (ndims(otf) > 2)
		  outSize = outSize*ones(1,ndims(otf));
	  else
		  [size_psf,idx] = min(size(otf));		  
		  if (size_psf > 1)
			  outSize = outSize*ones(1,2);
		  else
			  if (idx == 1)
				  outSize = [1 outSize];
			  else
				  outSize = [outSize 1];
			  end
		  end
	  end
  end

end;

% Check validity of the input parameters
% otf can be empty. it treats empty array as the fftn does
if ~isnumeric(otf) || issparse(otf),
  eid = sprintf('Images:%s:invalidOTF',mfilename);
  msg = 'Invalid OTF class: must be numeric, non-sparse.';
  error(eid,'%s',msg);
else
  otf = double(otf);
  if ~all(isfinite(otf(:))),
    eid = sprintf('Images:%s:otfContainsInfs',mfilename);
    msg = 'OTF must consist of finite values.';
    error(eid,'%s',msg);
  end
end
otfSize = size(otf);

% outSize:

if nargin==1,
  outSize = otfSize;% by default
elseif ~isa(outSize, 'double')
  eid = sprintf('Images:%s:invalidType',mfilename);
  msg = 'OUTSIZE has to be of class double.';
  error(eid,'%s',msg);
elseif any(outSize<0) | ~isreal(outSize) |...
    all(size(outSize)>1)| ~all(isfinite(outSize)),
  eid = sprintf('Images:%s:invalidOutSize',mfilename);
  msg = 'OUTSIZE has to be a vector of real, positive integers.';
  error(eid,'%s',msg);
end

if isempty(outSize),
  outSize = otfSize;
elseif ~isempty(otf),% empty arrays are treated similar as in the fftn
  [otfSize, outSize] = padlength(otfSize, outSize(:).');
  if any(outSize > otfSize)
    eid = sprintf('Images:%s:outSizeIsGreaterThanOtfSize',mfilename);
    msg1 = 'OUTSIZE must not exceed the size of the OTF array ';
    msg2 = 'in any dimension.';
    error(eid,'%s%s',msg1,msg2);
  end
end
