function info = get_hcday_builder_bbox_info(filename)
% info = get_hcday_builder_bbox_info(filename) returns 
% various informations about an HDF5 file created by hcday_builder_bbox:
% * version: file format version
% * description: description of the irradiation dataset
% * size: size of the irradiation dataset
% * time_offset = 1 (ie time 0 unused) for version 1 daily hcday_builder_bbox file with duration = 32 days, else 0
% * unit defines the irradiation unit wrt Wh/m2, e.g. unit = 1/10 means integer values expressed in 1/10th of Wh/m2

header = h5info(filename);

% find irradiation dataset
nb_datasets = size(header.Datasets, 1);
for i = 1:nb_datasets
    if (strcmp(header.Datasets(i).Name, 'set'))
        irradiation_dataset = header.Datasets(i);
    end
end

% size of the dataset
info.size = irradiation_dataset.Dataspace(1).Size;

% If version >= 2, the file format version is stored in 'file_format_version' attribute
if (size(header.Attributes, 1) >= 1)
    info.version = h5readatt(filename,'/','file_format_version');
elseif (info.size(3) == 32)
    info.version = 1.0; % daily hcday_builder_bbox file with duration = 32 days
elseif (info.size(3) == 24)
    info.version = 2.0; % hourly day-profile file with duration = 24 hours
else
    error('HDF5 file %s was not created by hcday_builder_bbox', filename);
end

% If version >= 3, the description of the irradiation dataset is stored in 'description' attribute of dataset 'set'.
if (info.version >= 3)
    info.description = h5readatt(filename,'/set','description');
end

% time_offset = 1 (ie time 0 unused) for version 1 daily hcday_builder_bbox file with duration = 32 days, else 0.
if (info.size(3) == 32)
    info.time_offset = 1;
else
    info.time_offset = 0;
end

% If version >= 2, the unit is stored in 'unit' attribute of dataset 'set'
if (info.version >= 2)
    info.unit = h5readatt(filename,'/set','unit');
elseif (info.size(3) == 32)
    info.unit = 1.0; % unit = Wh/m2 for daily irradiation
elseif (info.size(3) == 24)
    info.unit = 0.1 % unit = 0.1 Wh/m2 for hourly irradiation
else
    error('HDF5 file %s was not created by hcday_builder_bbox', filename);
end


