function [h,vx,hc] = histn(M,varargin)
%histn(M, ...) computes an histogram of the matrix M.
%
% histn(M, n) (n integer) computes an histogram with n buckets.
% histn(M, n1,... nN) (n1,... nN integers where N is the number of columns of M) computes an histogram with ni buckets for the column i.
% histn(M, v) (v vector) computes an histogram where v defines the buckets.
% histn(M, v1,... vN) (v1,... vN vectors where N is the number of columns of M) computes an histogram where vi defines the buckets for the column i.

[np,nd] = size(M);
p = 0.125;

% create matrix vx where each column vx{i} defines the buckets for the column i of M.
vx = cell(nd,1);
switch(length(varargin))
    case 1
        if (length(varargin{1})==1)
            % histn(M, n) (n integer) computes an histogram with n buckets.
            for kd = 1:nd,
                vx{kd} = linspace(percentile100(M(:,kd),p), percentile100(M(:,kd),100-p), varargin{1});
            end
        else
            % histn(M, v) (v vector) computes an histogram where v defines the buckets.
            for kd = 1:nd,
                vx{kd} = varargin{1}(:);
            end
        end
    case nd
        for kd = 1:nd,
            if (length(varargin{kd})==1)
                % histn(M, n1,... nN) (n1,... nN integers where N is the number of columns of M) computes an histogram with ni buckets for the column i.
                vx{kd} = linspace(percentile100(M(:,kd),p), percentile100(M(:,kd),100-p), varargin{kd});
            else
                % histn(M, v1,... vN) (v1,... vN vectors where N is the number of columns of M) computes an histogram where vi defines the buckets for the column i.
                vx{kd} = varargin{kd};
            end
        end
    otherwise
        error('histn: incorrect number of parameters');
end

x0dxnx = zeros(nd,3);
for kd = 1:nd,
    x0dxnx(kd,1) = vx{kd}(1);
    x0dxnx(kd,2) = vx{kd}(2)-vx{kd}(1);
    x0dxnx(kd,3) = length(vx{kd});
    err_max = max(abs(vx{kd}-(x0dxnx(kd,1)+[0:x0dxnx(kd,3)-1]*x0dxnx(kd,2))));
    if (err_max > 1e-6)
        error('Problem');
    end
end

Mk = zeros(np,nd);
for kd = 1:nd,  
    Mk_kd = round((M(:,kd)-x0dxnx(kd,1))/x0dxnx(kd,2))+1;
    Mk_kd((Mk_kd<1)|(Mk_kd>x0dxnx(kd,3)))=nan;
    Mk(:,kd) = Mk_kd;
end
Mk(any(isnan(Mk),2),:) = [];
npv = size(Mk,1);

Ck = cell(nd,1);
for kd = 1:nd
    Ck{kd} = Mk(:,kd);
end

sz = x0dxnx(:,3)';

h = zeros(sz);
vh = h(:);

ind = sub2ind(sz,Ck{:});

for kp = 1:npv,
   vh(ind(kp)) = vh(ind(kp)) + 1;
end

h = reshape(vh,sz);

if (nargout==0)&&(nd==2)
   s = (100*h'/sum(h(:)));
   imagesc(vx{1},vx{2},log10(s));
   map = jet(256);
   map(1,:) = [1 1 1];
   colormap(map);
   axis xy
   colorbar
   hold on
   [Mx1,Mx2] = ndgrid(vx{:});
   T = sum(Mx2.*h,2)./sum(h,2);
   plot(vx{1},T,'k','linewidth',3);
   V = sum(Mx2.^2.*h,2)./sum(h,2);
   plot(vx{1},T,'k','linewidth',3);
   plot(vx{1},sqrt(max(0,V-T.^2))+T,'k--');
   plot(vx{1},-sqrt(max(0,V-T.^2))+T,'k--');
   grid
   hold off
end
