function image_compare(varargin)
%image_compare Displays two 2D arrays for comparison.
%   image_compare(main_title, title1, M1, title2, M2) displays M1 and M2 2D arrays as images with the same scaling.  
%   The function also displays an image of differences (M2 - M1).
%   main_title will be the figure title (optional).
%
%   image_compare(main_title, title1, M1, title2, M2, mask), where mask is a 2D array of the same size, hides 
%   pixels with mask == 0.
%
%   image_compare(main_title, title1, M1, title2, M2, mask, borders), where borders = {x y} arrays, draws the (x, y) line segments. 
%
%   image_compare(main_title, title1, M1, title2, M2, mask, borders, clim, clim2) displays M1 and M2 with scaling = clim,
%   and the image of differences with scaling = clim2.

if (nargin < 4); error('image_compare: wrong parameters'); end

% main_title parameter is optional
if ~ischar(varargin{2})
    image_compare('', varargin{:})
    return
end
    
assert(nargin >= 5);

% figure title
main_title = varargin{1};
if ~ischar(main_title); error('image_compare: wrong parameter main_title'); end

% image and title 1
title1 = varargin{2};
if ~ischar(title1); error('image_compare: wrong parameter title1'); end
M1 = varargin{3};
if (length(size(M1)) ~= 2); error('image_compare: wrong parameter M1'); end

% image and title 2
title2 = varargin{4};
if ~ischar(title2); error('image_compare: wrong parameter title2'); end
M2 = varargin{5};
if (length(size(M2)) ~= 2); error('image_compare: wrong parameter M2'); end

% this function requires images type = double
M1 = double(M1);
M2 = double(M2);

% compute image of differences
diff = M2 - M1;

% mask
if (nargin >= 6)
    mask = varargin{6};
    M1(mask==0)=nan; % hide pixels with mask == 0
    M2(mask==0)=nan; % hide pixels with mask == 0
    diff(mask==0)=nan; % hide pixels with mask == 0
else
    mask = [];
end

% borders
if (nargin >= 7)
    borders = varargin{7};
    if ~iscell(borders); error('imagesc1: wrong parameter borders'); end
else
    borders = {};
end

% create alpha mask
alphaval = ones(size(M1)); 
alphaval(isnan(M1) | isnan(M2)) = 0.5;

% scaling for M1 + M2
if (nargin >= 8)
    clim = varargin{8};
else
    % compute scaling for M1 + M2, excluding outliers
    M1M2 = [M1 M2];
    min_M = percentile100(M1M2(:),0.1); % nanmin(M1M2(:));
    max_M = percentile100(M1M2(:),99.9); % nanmax(M1M2(:));
    if max_M == min_M % avoid division by zero
        max_M = min_M + 1.0;
    end
    clim = [min_M max_M];
end

% scaling for image of differences
if (nargin >= 9)
    clim2 = varargin{9};
else
    % compute scaling for image of differences
    min_M = percentile100(diff(:),0.1); % diff(M1M2(:));
    max_M = percentile100(diff(:),99.9); % diff(M1M2(:));
    if max_M == min_M % avoid division by zero
        max_M = min_M + 1.0;
    end
    clim2 = [min_M max_M];
end

% get current figure or create a new one
f = gcf;
figure(f)

% set window position
set(f,'Position', [200, 200, 550*2, 440*2]) % left, bottom, width, height

% turn off Latex text interpreter
set(f, 'DefaultTextInterpreter', 'none')

% draw M1
subplot(2,2,1)
if (isnan(clim(1)) | isnan(clim(2)))
    imagesc(M1)
else
    imagesc(M1, clim)
end
alpha(alphaval) % add alpha channel
hold on
if ~isempty(borders)
    plot(borders{1}, borders{2}, 'w') % draw borders
end
title(title1); % set subplot title
colorbar % show palette

% draw M2
subplot(2,2,2)
if (isnan(clim(1)) | isnan(clim(2)))
    imagesc(M2)
else
    imagesc(M2, clim)
end
alpha(alphaval) % add alpha channel
hold on
if ~isempty(borders)
    plot(borders{1}, borders{2}, 'w') % draw borders
end
title(title2); % set subplot title
colorbar % show palette

% draw image of differences
subplot(2,2,3)
if (isnan(clim2(1)) | isnan(clim2(2)))
    imagesc(diff)
else
    imagesc(diff, clim2)
end
alpha(alphaval) % add alpha channel
hold on
if ~isempty(borders)
    plot(borders{1}, borders{2}, 'w') % draw borders
end
title('Difference (right - left)'); % set subplot title
colorbar % show palette

% draw scaterplot and print statistics
fprintf('Statistics for %s wrt %s:\n', title2, title1);
subplot(2,2,4)
statd_irrad_v6(M2, M1, mask, 1, 2, '', {title2 title1}, clim);
title('Statistics (right wrt left)'); % set subplot title
    
% set figure and window title
mtit(main_title, 'FontSize',14, 'Color','k', 'Position',[0.5 1.05 0.5]);
set(f, 'name', main_title);

% force figure drawing now to workaround issues when opening many figures quickly
refreshdata(f)

