function [Ii, Ii_KT] = interp_I(I, H0, H0i, dt, dti, method)
%interp_I 1-D interpolation of an irradiation time serie
%
%   [Ii, Ii_KT] = interp_I(I,H0,HOi,dt,dti) interpolates to find Ii, the values of the
%   underlying function I=F(t) at the query points ti = [dti:dti:24]. 
%   dt and dti are the in and out time steps.
%
%   H0 is the Top of Atmosphere irradiation time serie corresponding to I, H0i to Ii.
%
%   The function computes also Ii_KT = ???
%
%   [Ii, Ii_KT] = interp_I(I,H0,HOi,dt,dti,method) specifies alternate methods.
%   The default is linear interpolation. 
%   Available methods are:
%     'nearest'  - nearest neighbor interpolation
%     'linear'   - linear interpolation
%     'spline'   - piecewise cubic spline interpolation (SPLINE)
%     'pchip'    - shape-preserving piecewise cubic interpolation
%     'cubic'    - same as 'pchip'
%     'v5cubic'  - the cubic interpolation from MATLAB 5, which does not
%                  extrapolate and uses 'spline' if t is not equally spaced.

if (nargin < 6)
    method = 'linear';
end

t = [dt:dt:24];
nt = length(t);
ti = [dti:dti:24];
nti = length(ti);

nd = size(I,2);
if (size(I,1) ~= nt)
    error('I pas de la bonne taille');
end
if (size(H0,1) ~= nt)
    error('H0 pas de la bonne taille');
end
if (size(H0i,1) ~= nti)
    error('H0i pas de la bonne taille');
end

si2 = dt/dti;
Hi2 = sparse(nt, nti);
for k = 2:nt,
    Hi2(k,k*si2+[-si2+1:0]) = 1;
end
Ker_Hi2 = null(full(Hi2));
pi_Hi2 = pinv(full(Hi2));
Ri = Ker_Hi2*Ker_Hi2';

minH0 = 5*dt;
minH0i = minH0*dti/dt;

KT = zeros(nt,nd);
J = H0>minH0; 
KT(J) = I(J)./H0(J);

KTi = min(1,max(0,interp1(t-dt/2, KT, ti-dti/2, method, 0))).*(H0i>minH0i);
Ii_KT = KTi.*H0i;

Ii_MPq = pi_Hi2*I;
Ii = Ii_MPq + Ri*(Ii_KT-Ii_MPq);


