function [Ii, KTi, KT] = interp_Irrad(I, H0, H0i, t, ti, method)
%interp_Irrad 1-D interpolation of an irradiation time serie
%
%   [Ii, KTi, KT] = interp_Irrad(I,H0,HOi,t,ti) interpolates to find Ii, the values of the
%   underlying function I=F(t) at the query points ti. t must
%   be a vector of length N.
%   If I is a vector, then it must also have length N, and Ii is the
%   same size as ti.  If I is an array of size [N,D1,D2,...,Dk], then
%   the interpolation is performed for each D1-by-D2-by-...-Dk value
%   in I(i,:,:,...,:).
%   If ti is a vector of length M, then Ii has size [M,D1,D2,...,Dk].
%   If ti is an array of size [M1,M2,...,Mj], then Ii is of size
%   [M1,M2,...,Mj,D1,D2,...,Dk].
%
%   H0 is the Top of Atmosphere irradiation time serie corresponding to I, H0i to Ii.
%
%   The function computes also KT = I/H0 and KTi = Ii/H0i.
%
%   [Ii, KTi, KT] = interp_Irrad(I,H0,HOi,t,ti,method) specifies alternate methods.
%   The default is linear interpolation. 
%   Available methods are:
%     'nearest'  - nearest neighbor interpolation
%     'linear'   - linear interpolation
%     'spline'   - piecewise cubic spline interpolation (SPLINE)
%     'pchip'    - shape-preserving piecewise cubic interpolation
%     'cubic'    - same as 'pchip'
%     'v5cubic'  - the cubic interpolation from MATLAB 5, which does not
%                  extrapolate and uses 'spline' if t is not equally spaced.

if (nargin < 6)
    method = 'linear';
end

nt = length(t);
nti = length(ti);
dt = t(2)-t(1);
dti = ti(2)-ti(1);

nd = size(I,2);
if (size(I,1) ~= nt)
    error('I pas de la bonne taille');
end
if (size(H0,1) ~= nt)
    error('H0 pas de la bonne taille');
end
if (size(H0i,1) ~= nti)
    error('H0i pas de la bonne taille');
end

si2 = dt/dti;
Hi2 = sparse(nt, nti);
for k = 2:nt,
    %     Hi2(k,k*si2+[-si2+1:0]) = 1;
    idx_i = find((ti>t(k-1))&(ti<=t(k)));
    Hi2(k,idx_i) = 1;
end
Ker_Hi2 = null(full(Hi2));
pi_Hi2 = pinv(full(Hi2));
Ri = Ker_Hi2*Ker_Hi2';

minH0 = 10*dt;
minH0i = minH0*dti/dt;

KT = zeros(nt,nd);
KT(H0>minH0) = I(H0>minH0)./H0(H0>minH0);

KTi = max(0,interp1(t-dt/2, KT, ti-dti/2, method, 0)).*(H0i>minH0i);
Ii_KT = KTi.*H0i;

% EW 2014-10-16: Workaround bug when interpoling from 10 to 1 minutes
% Ii_MPq = pi_Hi2*I;
% Ii = Ii_MPq + Ri*(Ii_KT-Ii_MPq);
Ii = Ii_KT;
