function [lig, col, lig_fixed, col_fixed] = latlon_to_ligcol(lat, lon)
%Convert geographical coordinates (degrees, positive NE and negative SW)
%to pixel (lig, col) in HRIT MSG image.
%
% [lig, col] is the pixel using Helioclim 3 formula. There is a 0.5 pixel shift compared to
% [lig_fixed, col_fixed] which uses the formula in:
%   [1] LRIT/HRIT Global Specification, Issue 2.6, August 1999.
%
%The returned value return1 is :
%1 in case of wrong input geographical coordinates,
%3 in case of non usable pixel,
%0 if OK, and when different from 0 lig and col are set to -999.

%R = 6371.0;         %/* Earth radius in km */
%Io = 1367.0;         %/* solar constant in W/m2 */
Rp = 6356.580;    %/* earth polar radius cf.LRIT/HRIT Global specif. */
Re = 6378.170;    %/* earth equatorial radius .......................*/
Hs = 35785.80;%    /* altitude of satellite   */
Lon0 = 0.0;    %   /* longitude of satellite in degrees */
Scanstep = 0.00025153;%/* instrument scan step angle (rd) for 3 lines */
nblig = 3712; %/* nblig in 3 km image */
nbcol = 3712; %/* nbcol .............................  */
%I10_NBLIG = 1238; %/* nblig in reduced resolution image (10 km) */
%I10_NBCOL = 1238; %/* nbcol ...........................*/
deg2rad = pi()/180;
rad2deg = 180/pi();
deltaX = Scanstep/3.0*rad2deg; %/* 1 pixel scanned /col  */
deltaY = Scanstep/3.0*rad2deg; %/* ............... /line */

lig = -999;
col = -999;
if(lat < -90.0 || lon < -90),  return1 = 1;
end
if(lat >  90.0 || lon >  90), return1 = 1;
end
tlat   = lat * deg2rad ;
tlon   = lon * deg2rad ;
coslat = cos(tlat) ;
sinlat = sin(tlat) ;
coslon = cos(tlon) ;

ReRp = Re * Rp ;
Hs2  = Hs * Hs ;
Rp2  = Rp * Rp ;
Re2  = Re * Re ;
rom  = ReRp / sqrt(Rp2 * coslat * coslat + Re2 * sinlat * sinlat) ;
rom2 = rom * rom ;
y2   = Hs2 + rom2 - 2 * Hs * rom * coslat * coslon ;
r1   = y2 + rom2 ;
if (r1 > Hs2),
    return1 = 3;
end

RpdRe   = Rp / Re ;
Rs      = Re + Hs ;
tanlat  = tan(tlat) ;
sinlon  = sin(tlon) ;
tlon0   = Lon0 * deg2rad ;
sinlon0 = sin(tlon0) ;
coslon0 = cos(tlon0) ;
teta    = atan(RpdRe * tanlat) ;
costeta = cos(teta) ;
sinteta = sin(teta) ;
xt = Re * costeta * coslon ;
yt = Re * costeta * sinlon ;
zt = Rp * sinteta ;
wt = xt - Rs * coslon0 ;
vt = yt - Rs * sinlon0 ;
px = atan(( coslon0 * vt - wt * sinlon0 ) / ( sinlon0 * vt + wt * coslon0) ) ;
py = atan(( zt * (tan(px) * sinlon0 - coslon0) / wt ) * cos(px)) ;
px = px / deg2rad ;
py = py / deg2rad ;

xxr = px / deltaX ;%/* coordinates centered over subsatellite point */
yyr = py / deltaY ;

xr = floor(px/deltaX)+0.5;
yr = floor(py/deltaY)+0.5;

col = floor(xr+(nbcol/2)+0.5);%(int)
lig = floor(yr+(nblig/2)+0.5);%(int)
lig = nblig - lig +1 ; %/* reverse image N-S */
col = nbcol - col +1 ;

col_fixed = nbcol - (xxr+nbcol/2) + 1;
lig_fixed = nblig - (yyr+nblig/2) + 1;

return1 = 0;
