function [lat, lon, return1] = ligcol_to_latlon(lig, col)
%   Provides for any pixel (lig, col) in HRIT MSG image
%   its geographical coordinates (degrees, positive NE and negative SW).
%
% This is the formula used by Helioclim 3. There is a 0.5 pixel shift compared to:
%   [1] LRIT/HRIT Global Specification, Issue 2.6, August 1999.
%
%   The returned value return1 is :
%   1 in case of wrong input data (i.e. lig and col outside the image frame),
%   3 in case of non usable pixel,
%   0 if OK, and when different from 0 lat and lon are set to -999.

Rp = 6356.580;    %/* earth polar radius cf.LRIT/HRIT Global specif. */
Re = 6378.170;    %/* earth equatorial radius .......................*/
Hs = 35785.80;%    /* altitude of satellite   */
Lon0 = 0.0;    %   /* longitude of satellite in degrees */
Scanstep = 0.00025153;%/* instrument scan step angle (rd) for 3 lines */
nblig = 3712; %/* nblig in 3 km image */
nbcol = 3712; %/* nbcol .............................  */

return1= 0;

deg2rad = pi/180;
rad2deg = 180/pi;
deltaX = Scanstep/3.0*rad2deg; %/* 1 pixel scanned /col  */
deltaY = Scanstep/3.0*rad2deg; %/* ............... /line */

Rs     = Re + Hs ;      % /* orbit radius */
yk2    = Rs * Rs / (Re * Re) ;
A      = 1.0/297.0;      %/* earth oblatness */

lat = -999.0;
lon = -999.0;
if (lig < 1 || lig > nblig) return1= 1
end
if (col < 1 || col > nbcol) return1= 1
end

X = nbcol - col + 1;  %/* reverse image from NW-SE to SE-NW acquisition*/
Y = nblig - lig + 1;
X = X - (nbcol/2) - 0.5;%/* translate subsatellite point to (0,0) */
Y = Y - (nblig/2) - 0.5;
X = X * deltaX * deg2rad;
Y = Y * deltaY * deg2rad;
cosX   = cos(X);
tanX   = tan(X);
tan2X  = tanX*tanX;
tanY   = tan(Y);
tan2Y  = tanY*tanY;

tlon0   = Lon0 * deg2rad; %/* longitude of satellite */
coslon0 = cos(tlon0);
sinlon0 = sin(tlon0);

val1=1.0 + tan2X;
val2=1.0 + tan2Y*(A+1)*(A+1);

vmu = yk2 - (yk2 - 1)*val1*val2;
if( vmu < 0 ) 
    return1= 3; %/* exclude a negative argument for sqrt() */
end
vmu = ( Rs - Re * sqrt(vmu) ) / (val1 * val2);
xt = (Rs * coslon0) + ( vmu * (tanX * sinlon0 - coslon0) );
yt = (Rs * sinlon0) - ( vmu * (tanX * coslon0 + sinlon0) );
zt = vmu * tanY / cosX;
teta=asin(zt / Rp);

lat = atan( tan(teta) * Re / Rp) * rad2deg;
lon = atan(yt/xt) * rad2deg;
