function [I,pct,pce] = mean_I_gf(I_i, R_i, J_i, t_i, t, eps_dt)
% [I,pct,pce] = mean_I_gf(I_i, R_i, J_i, t_i, t, eps_dt)
% computes the mean of the irradiances I_i at time steps t_i for time steps t.
% NaN values are interpolated thanks to clear sky irradiances R_i.
%
% Parameters:
% I_i: irradiances at time steps t_i
% R_i: clear sky irradiances
% J_i: time steps to consider (logical)
% t_i: input time steps
% t  : output time steps
%
% Output:
% I  : irradiances at time steps t
% pct: percentage of non-NaN input time steps in J_i
% pce: R_i mean for non-NaN input time steps / R_i mean for input time steps

if (length(t)>=2)
    dt = t(2)-t(1);
    if (nargin < 6)
        eps_dt = dt/100;
    end
end

nt = length(t); % number of steps per day
nd = size(I_i,2); % number of days

% init output variables
I = zeros(nt,nd);
pct = ones(nt,nd);
pce = ones(nt,nd);

% for each output interval K = [k-1, k] of each day
for k = 1:nt
    % get input time steps in K
    if k == 1
        % special case: first interval K = [0, 1] of each day
        ind_i = find(t_i<=t(k));
    else
        ind_i = find((t_i>t(k-1)+eps_dt)&(t_i<=t(k)+eps_dt));
    end
    
     % get input data in interval K
    I_k = I_i(ind_i,:);
    R_k = R_i(ind_i,:);
    J_k = J_i(ind_i,:);
    
    % get non-NaN input time steps (ie irradiance not NaN) in J_i
    ok_k = (~isnan(I_k)) & (J_k==1); 
    nok_k = sum(ok_k,1); % number of non-NaN input time steps in J_i
    nok_max = sum(double(J_k==1),1); % number of input time steps in J_i
    
    % percentage of non-NaN input time steps in J_i
    pct_k = pct(k,:);
    pct_k(nok_max>0)=nok_k(nok_max>0)./nok_max(nok_max>0); 
    pct(k,:) = pct_k;
    
    % compute I_i mean in K
    I_k(ok_k==0) = 0;
    nanmean_I_k = sum(I_k,1)./nok_k; 
    
    % compute R_i mean for non-NaN input time steps in J_i
    R_k_nan = R_k;
    R_k_nan(ok_k==0) = 0;
    R_k_nan(J_k==0) = 0;
    nanmean_R_k_nan = sum(R_k_nan,1)./nok_k; 
    
    % compute R_i mean for input time steps in J_i
    R_k(J_k==0) = 0;
    mean_R_k = sum(R_k,1)./nok_k; 
    
    % compute pce = R_i mean for non-NaN input time steps / R_i mean for input time steps
    pce_k = pce(k,:);
    pce_k(mean_R_k>0) = nanmean_R_k_nan(mean_R_k>0)./mean_R_k(mean_R_k>0);
    pce(k,:) = pce_k;
    
    % compute output irradiation for interval K
    I(k,:) = nanmean_I_k./pce_k; 
end

