function H = ri2ftm(varargin)

[psf, psfSize, outSize] = ParseInputs(varargin{:});
% if (ndims(psf) == 3)
%     H = zeros(outSize(1),outSize(2),psfSize(3));
%     for k = 1:psfSize(3),
%         H(:,:,k) = ri2ftm(psf(:,:,k),outSize(1:2));
%     end
%     return;
% end

if  ~all(psf(:)==0),
   
   % Pad the PSF to outSize
   padSize = outSize - psfSize;
   psf     = pad(psf, padSize, 'post');

   % Circularly shift otf so that the "center" of the PSF is at the
   % (1,1) element of the array.
   psf    = circshift(psf,-floor(psfSize/2));

   % Compute the OTF
   otf = fftn(psf);

   % Estimate the rough number of operations involved in the 
   % computation of the FFT.
   nElem = prod(psfSize);
   nOps  = 0;
   for k=1:ndims(psf)
      nffts = nElem/psfSize(k);
      nOps  = nOps + psfSize(k)*log2(psfSize(k))*nffts; 
   end

   % Discard the imaginary part of the psf if it's within roundoff error.
   if max(abs(imag(otf(:))))/max(abs(otf(:))) <= nOps*eps
      otf = real(otf);
   end
else
   otf = zeros(outSize);
end

H = fftshift(otf);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Parse inputs
%%%

function [psf, psfSize, outSize] = ParseInputs(varargin)

% error(nargchk(1,2,nargin,'struct'));
switch nargin
case 1       % PSF2OTF(PSF) 
  psf = varargin{1};   
case 2       % PSF2OTF(PSF,OUTSIZE) 
  psf = varargin{1}; 
  outSize = varargin{2};
  if (length(outSize)==1)
	  if (ndims(psf) > 2)
		  outSize = outSize*ones(1,ndims(psf));
	  else
		  [size_psf,idx] = min(size(psf));		  
		  if (size_psf > 1)
			  outSize = outSize*ones(1,2);
		  else
			  if (idx == 1)
				  outSize = [1 outSize];
			  else
				  outSize = [outSize 1];
			  end
		  end
	  end
  end
end;

% Check validity of the input parameters
% psf can be empty. it treats empty array as the fftn does
if ~isnumeric(psf) || issparse(psf)
  eid = sprintf('Images:%s:expectedNonSparseAndNumeric',mfilename);
  msg = 'Invalid PSF class: must be numeric, non-sparse.';
  error(eid,'%s',msg);
else
  psf = double(psf);
  if ~all(isfinite(psf(:)))
    eid = sprintf('Images:%s:expectedFinite',mfilename);
    msg = 'PSF must consist of finite values.';
    error(eid,'%s',msg);
  end
end
psfSize = size(psf);

% outSize:
if nargin==1,
  outSize = psfSize;% by default
elseif ~isa(outSize, 'double')
  eid = sprintf('Images:%s:invalidType',mfilename);
  msg = 'OUTSIZE has to be of class double.';
  error(eid,'%s',msg);
elseif any(outSize<0) | ~isreal(outSize) |...
    all(size(outSize)>1) | ~all(isfinite(outSize))
  eid = sprintf('Images:%s:invalidOutSize',mfilename);
  msg = 'OUTSIZE has to be a vector of real, positive integers.';
  error(eid,'%s',msg);
end

if isempty(outSize),
  outSize = psfSize;
elseif ~isempty(psf),% empty arrays are treated similar as in the fftn
  [psfSize, outSize] = padlength(psfSize, outSize(:).');
  if any(outSize < psfSize)
    eid = sprintf('Images:%s:outSizeIsSmallerThanPsfSize',mfilename);
    msg1 = 'OUTSIZE cannot be smaller than the PSF array size in any ';
    msg2 = 'dimension.';
    error(eid,'%s%s',msg1,msg2);
  end
end

