function d = statd_irrad_v2_plus(mG, mR, mCond, reg, graph_scatter, unit, labels, clim)
%statd_irrad_v2_plus(mG, mR, mCond, reg, graph_scatter, unit, labels, clim) computes statistics like bias, RMSE, correlation coefficient... to compare 2 series.
%
%   statd_irrad_v2_plus(mG, mR) compares the mG serie with the reference serie mR.
%
%   mCond parameter restricts the comparison to elements such that mCond is true.
%
%   reg parameter computes a regression. 
%   reg may have a value from 1 to 5 meaning '1 0', 'Regress[a,b]', 'Regress[a,0]', 'RobustFit[a,b]', 'PCA[a,b]'
%
%   Set graph_scatter = 1 to draw a comparison graph in the current figure.
%   Set graph_scatter = 2 to draw a comparison graph in the current sub-figure.
%
%   'unit' parameter defines the data unit (none by default).
%
%   labels = {ylabel xlabel} defines the axis labels ('Estimation' and 'Reference' by default).
%
%   clim = [min max] can specify the scaling.

% init d structure.
% note: statd_irrad_v6() and statd_irrad_v2_plus() share the same return type.
fn = {'n','r','numr','nomr','mean_ref','ve_ref','std_ref','bias','mbe','mae',...
    'stdev','rmse','bias_r','mbe_r','mae_r','stdev_r','rmse_r','cc','KSI','KSI_r','p90','p90_r','p95','p95_r','p99p7','p99p7_r'};
d.n = 0;
d.r = [nan nan]; % regression coefficients
d.numr = nan;
d.nomr = '';
for k = 5:length(fn)
    d.(fn{k}) = nan;
end

% condition?
if (nargin < 3)
    mCond = [];
end
% regression?
if (nargin < 4)
    reg = 1;
end
% draw graph?
if (nargin < 5)
    graph_scatter = 0;
end
% unit?
if (nargin < 6)
    unit = '';
end
% labels?
if (nargin < 7)
    labels = {'Estimation','Reference'};
end
% scaling?
if (nargin < 8)
    clim = [];
end

nd = size(mG,1);

% Convert params to expected types
mG = double(mG);
mR = double(mR);
mCond = logical(mCond);

% flatten data
mG = mG(:);
mR = mR(:);

if (isempty(mCond))
    mCond = true(size(mR));
end
if (length(mCond)==1)
    mCond = repmat(mCond,size(mG));
end
mCond = mCond(:);

% find elements not nan/infinity
ind = find(not(isnan(mG)|isnan(mR)|isinf(mR)|isinf(mG)));
if isempty(ind)
    d.n = 0;
    d.r = [nan nan];
    d.numr = reg;
    d.nomr = '';
    if (nargout == 0)
        fprintf(1,'NDATA:\t  %d\n', d.n);
        fprintf(1,'\n');
    end
    return;
end

mG = mG(ind);
mR = mR(ind);
mCond = mCond(ind);

vG = mG(mCond ~= 0); % elements not nan/infinity and not filtered out
vR = mR(mCond ~= 0);

d.n = length(vG); % #elements not nan/infinity and not filtered out
if d.n==0
    d.n = 0;
    d.r = [nan nan];
    d.numr = reg;
    d.nomr = '';
    if (nargout == 0)
        fprintf(1,'NDATA:\t  %d\n', d.n);
        fprintf(1,'\n');
    end
    return;
end

% compute regression
nr = 5;
cr = nan(nr,2); % regression coefficients
switch(reg)
    case 1
        cr(1,:) = [1 0];
    case 2
        cr(2,:) = [regress(vR, [vG ones(d.n,1)])]';
    case 3
        cr(3,:) = [regress(vR, [vG]) 0];
    case 4
        bls = robustfit(vG, vR);
        cr(4,:) = fliplr(bls');
    case 5
        Z = [vG-mean(vG) vR-mean(vR)];
        [U,S,V] = svd(Z,'econ');
        cr(5,:) = [-V(1,2)/V(2,2) mean(vR)+mean(vG)*V(1,2)/V(2,2)];
end
nom_cr = {'1 0','Regress[a,b]','Regress[a,0]','RobustFit[a,b]','PCA[a,b]'};
d.r = cr(reg,:);
d.numr = reg;
d.nomr = nom_cr{reg};
vGr = polyval(d.r,vG);

mean_r = mean(vR); % absolute mR mean
d.mean_ref = mean_r;
d.ve_ref = sqrt(mean(vR.^2));
d.std_ref = std(vR);
d.bias = mean(vGr-vR); % bias of mG wrt mR
d.mbe = d.bias;
d.mae = mean(abs(vGr-vR)); % absolute bias of mG wrt mR
var = mean((vGr-vR).^2); % variance
d.stdev = sqrt(var - d.bias^2); % standard deviation
d.rmse = sqrt(var); % RMSE

% same statistics expressed as percentages of absolute mR mean
d.bias_r = 100*d.bias/mean_r;
d.mbe_r = d.bias_r;
d.mae_r = 100*d.mae/mean_r;
d.stdev_r = 100*d.stdev/mean_r;
d.rmse_r = 100*d.rmse/mean_r;

% correlation coefficient
if (length(vGr) >= 2)
    C = corrcoef(vGr,vR);
    d.cc = C(1,2);
else
    d.cc = nan;
end

if (d.n >= 100)

    % Test KSI

    nb = min(d.n,200);
    xmin = min([vR ; vGr]);
    xmax = max([vR ; vGr]);
    dx = (xmax-xmin)/(nb-1);
    vx = xmin + [0:nb-1]*dx;

    hcR = zeros(nb,1);
    hcG = zeros(nb,1);

    for k = 1:nb,
        ind = find((vR <= xmin + (k-1)*dx));
        hcR(k) = length(ind)/d.n;
    end

    for k = 1:nb,
        ind = find((vGr <= xmin + (k-1)*dx));
        hcG(k) = length(ind)/d.n;
    end

    D = abs(hcG-hcR);
    d.KSI = sum(D)*dx;

    Vc = 1.63/sqrt(d.n);
    a_c = Vc*(xmax-xmin);

    d.KSI_r = 100*d.KSI/a_c;
else
    d.KSI = nan;
    d.KSI_r = nan;
end

% prepare unit to print it
if ~strcmp(unit, '')
    unit_par = [' (' unit ')']; % add space prefix + parenthesis
    unit = [' ' unit]; % add space prefix
else
    unit_par = '';
end
    
if (nargout == 0)
    if reg>1
        fprintf(1,'TYPE REG: %s (#%d)\n', d.nomr, d.numr);
        fprintf(1,'COEF REG: [%.2f, %.2f]\n', d.r(1), d.r(2));
    end
    fprintf(1,'NDATA:    %d\n', d.n);
    fprintf(1,'MEAN:     %.2f%s\n', d.mean_ref, unit);
    fprintf(1,'Bias:     %0.2f %% (%.2f%s)\n', d.bias_r, d.bias, unit);
    fprintf(1,'MAE:      %0.2f %% (%.2f%s)\n', d.mae_r, d.mae, unit);
    fprintf(1,'STDE:     %0.2f %% (%.2f%s)\n', d.stdev_r, d.stdev, unit);
    fprintf(1,'RMSE:     %0.2f %% (%.2f%s)\n', d.rmse_r, d.rmse, unit);
    if (d.n >= 10)
        fprintf(1,'CC:       %0.3f\n', d.cc);
        fprintf(1,'KSI:      %05.2f %% (%.2f)\n', d.KSI_r, d.KSI);
    end
    fprintf(1,'\n');
end

hsign = inline('char(43+2*(sign(x)==-1))','x');

% display scatter plot
if (graph_scatter>0)
    
    f = gcf; % get current figure or create a new one
    
    % if graph_scatter == 1, draw a comparison graph in the current figure
    if graph_scatter == 1
        if strcmp(get(f,'visible'),'on') % if figure is visible
            figure(f) % bring figure to front
        end
        clf % clear figure
        set(f,'position',[100 50 800 675]) % set position and size
    end
    
    % clim parameter can specify the scaling
    if ~isempty(clim)
        xmin = clim(1);
        xmax = clim(2);
        ymin = clim(1);
        ymax = clim(2);
    else
        % LS 11/2017: compute min and max, excluding outliers
        xmin = percentile100(vR(:),0.03); % min(vR(:));
        xmax = percentile100(vR(:),99.97); % max(vR(:));
        ymin = percentile100(vGr(:),0.03); % min(vGr(:));
        ymax = percentile100(vGr(:),99.97); % max(vGr(:));
    end
    vmin = min(xmin,ymin);
    vmax = max(xmax,ymax);
    dmax = vmax-vmin;
    
    hp = plot(vR, vGr, 'r*', ...
        [vmin vmax], [vmin vmax], 'k--');
    axis([vmin vmax vmin vmax])
    grid
    %     legend(hp,sprintf('(ref,estimation+%s)',nom_cr{reg}),'x = y','location','best')
    xlabel([labels{2} unit_par])
    if (d.numr == 1)
        ylabel([labels{1} unit_par])
    elseif (d.numr == 2)
        ylabel(sprintf('%.2f * %s%s + %.2f', d.r(1), labels{1}, unit_par, d.r(2)))
    else
        ylabel(sprintf('%.2f * %s%s', d.r(1), labels{1}, unit_par))
    end
    
    % LS 09/2017: avoid crash if vmax-vmin < 0.5
    if vmax-vmin > 0.5
        set(gca,'Xtick',round(linspace(vmin,vmax,5)*10)/10,'Ytick',round(linspace(vmin,vmax,5)*10)/10);
    else
        set(gca,'Xtick',linspace(vmin,vmax,5),'Ytick',linspace(vmin,vmax,5));
    end
    
    if reg==1
        txt_stat{1} = sprintf('NDATA: %d', d.n);
        txt_stat{2} = sprintf('MEAN:  %.2f%s', d.mean_ref, unit);
        txt_stat{3} = sprintf('Bias:  %0.2f %% (%.2f%s)', d.bias_r, d.bias, unit);
        txt_stat{4} = sprintf('MAE:   %0.2f %% (%.2f%s)', d.mae_r, d.mae, unit);
        txt_stat{5} = sprintf('STDE:  %0.2f %% (%.2f%s)', d.stdev_r, d.stdev, unit);
        txt_stat{6} = sprintf('RMSE:  %0.2f %% (%.2f%s)', d.rmse_r, d.rmse, unit);
        if (d.n >= 10)
            txt_stat{7} = sprintf('CC:    %0.3f', d.cc);
        end
    else
        txt_stat{1} = sprintf('COEF REG: [%.2f, %.2f]', d.r(1), d.r(2));
        txt_stat{2} = sprintf('NDATA: %d', d.n);
        txt_stat{3} = sprintf('MEAN:  %.2f%s', d.mean_ref, unit);
        txt_stat{4} = sprintf('Bias:  %0.2f %% (%.2f%s)', d.bias_r, d.bias, unit);
        txt_stat{5} = sprintf('MAE:   %0.2f %% (%.2f%s)', d.mae_r, d.mae, unit);
        txt_stat{6} = sprintf('STDE:  %0.2f %% (%.2f%s)', d.stdev_r, d.stdev, unit);
        txt_stat{7} = sprintf('RMSE:  %0.2f %% (%.2f%s)', d.rmse_r, d.rmse, unit);
        if (d.n >= 10)
            txt_stat{8} = sprintf('CC:    %0.3f', d.cc);
        end
    end
    
    text(vmin+dmax/50,vmax-dmax/50,txt_stat,...
        'VerticalAlignment','top', 'FontName','Courier', 'FontSize',9, 'BackgroundColor',[1 1 1]*0.95);

    % force figure drawing now to workaround issues when opening many figures quickly
    refreshdata(f)
    
end
