function d = statd_irrad_v6(mG, mR, mCond, reg, graph_scatter, unit, labels, clim)
%statd_irrad_v6(mG, mR, mCond, reg, graph_scatter, unit, labels, clim) computes statistics like bias, RMSE, correlation coefficient... to compare 2 series.
%
%   statd_irrad_v6(mG, mR) compares the mG serie with the reference serie mR.
%
%   mCond parameter restricts the comparison to elements such that mCond is true.
%
%   reg parameter computes a regression. 
%   reg may have a value from 1 to 5 meaning '1 0', 'Regress[a,b]', 'Regress[a,0]', 'RobustFit[a,b]', 'PCA[a,b]'
%
%   Set graph_scatter = 1 to draw a comparison graph in the current figure.
%   Set graph_scatter = 2 to draw a comparison graph in the current sub-figure.
%
%   'unit' parameter defines the data unit (none by default).
%
%   labels = {ylabel xlabel} defines the axis labels ('Estimation' and 'Reference' by default).
%
%   clim = [min max] can specify the scaling.

% init d structure.
% note: statd_irrad_v6() and statd_irrad_v2_plus() share the same return type.
fn = {'n','r','numr','nomr','mean_ref','ve_ref','std_ref','bias','mbe','mae',...
    'stdev','rmse','bias_r','mbe_r','mae_r','stdev_r','rmse_r','cc','KSI','KSI_r','p90','p90_r','p95','p95_r','p99p7','p99p7_r'};
d.n = 0;
d.r = [nan nan]; % regression coefficients
d.numr = nan;
d.nomr = '';
for k = 5:length(fn)
    d.(fn{k}) = nan;
end

% condition?
if (nargin < 3)
    mCond = [];
end
% regression?
if (nargin < 4)
    reg = 1;
end
% draw graph?
if (nargin < 5)
    graph_scatter = 0;
end
% unit?
if (nargin < 6)
    unit = '';
end
% labels?
if (nargin < 7)
    labels = {'Estimation','Reference'};
end
% scaling?
if (nargin < 8)
    clim = [];
end

nd = size(mG,1);

% Convert params to expected types
mG = double(mG);
mR = double(mR);
mCond = logical(mCond);

% flatten data
mG = mG(:);
mR = mR(:);

if (isempty(mCond))
    mCond = ones(size(mR));
end
if (length(mCond)==1)
    mCond = repmat(mCond,size(mG));
end
mCond = mCond(:);

% find elements not nan/infinity
ind = find(not(isnan(mG)|isnan(mR)|isinf(mR)|isinf(mG)));
if isempty(ind)
    d.n = 0;
    d.r = [nan nan];
    d.numr = reg;
    d.nomr = '';
    if (nargout == 0)
        fprintf(1,'NDATA:\t  %d\n', d.n);
        fprintf(1,'\n');
    end
    return;
end

mG = mG(ind);
mR = mR(ind);
mCond = mCond(ind);

vG = mG(mCond ~= 0); % elements not nan/infinity and not filtered out
vR = mR(mCond ~= 0);

d.n = length(vG); % #elements not nan/infinity and not filtered out
if d.n==0
    d.n = 0;
    d.r = [nan nan];
    d.numr = reg;
    d.nomr = '';
    if (nargout == 0)
        fprintf(1,'NDATA:\t  %d\n', d.n);
        fprintf(1,'\n');
    end
    return;
end

% compute regression
nr = 5;
cr = nan(nr,2); % regression coefficients
switch(reg)
    case 1
        cr(1,:) = [1 0];
    case 2
        cr(2,:) = [regress(vR, [vG ones(d.n,1)])]';
    case 3
        cr(3,:) = [regress(vR, [vG]) 0];
    case 4
        bls = robustfit(vG, vR);
        cr(4,:) = fliplr(bls');
    case 5
        Z = [vG-mean(vG) vR-mean(vR)];
        [U,S,V] = svd(Z,'econ');
        cr(5,:) = [-V(1,2)/V(2,2) mean(vR)+mean(vG)*V(1,2)/V(2,2)];
end
nom_cr = {'1 0','Regress[a,b]','Regress[a,0]','RobustFit[a,b]','PCA[a,b]'};
d.r = cr(reg,:);
d.numr = reg;
d.nomr = nom_cr{reg};
vGr = polyval(d.r,vG);

mean_r = mean(vR); % absolute mR mean
d.mean_ref = mean_r;
d.ve_ref = sqrt(mean(vR.^2));
d.std_ref = std(vR);
d.bias = mean(vGr-vR); % bias of mG wrt mR
d.mbe = d.bias;
d.mae = mean(abs(vGr-vR)); % absolute bias of mG wrt mR
var = mean((vGr-vR).^2); % variance
d.stdev = sqrt(var - d.bias^2); % standard deviation
d.rmse = sqrt(var); % RMSE

% same statistics expressed as percentages of absolute mR mean
d.bias_r = 100*d.bias/mean_r;
d.mbe_r = d.bias_r;
d.mae_r = 100*d.mae/mean_r;
d.stdev_r = 100*d.stdev/mean_r;
d.rmse_r = 100*d.rmse/mean_r;

% correlation coefficient
if (length(vGr) >= 2)
    C = corrcoef(vGr,vR);
    d.cc = C(1,2);
else
    d.cc = nan;
end

if (d.n >= 35)
    
    % KSI
    
    s_vR = sort(vR);
    s_vGr = sort(vGr);
    [s_rg,idx1] = unique([s_vR;s_vGr]);
    xmin = s_rg(1);
    xmax = s_rg(end);
    pc = .05;
    xmin_p = percentile100([s_vR;s_vGr],pc);
    xmax_p = percentile100([s_vR;s_vGr],100-pc);
    P = length(s_rg);
    N = length(vR);
    CDF_r = zeros(P,1);
    CDF_g = zeros(P,1);
    
    %     CDF_r_2 = zeros(P,1);
    
    k_r = 1;
    k_g = 1;
    for p = 1:P,
        
        while 1
            if (k_r > N)
                break;
            end
            if (s_vR(k_r)>s_rg(p))
                break;
            end
            k_r = k_r + 1;
        end
        
        while 1
            if (k_g > N)
                break;
            end
            if (s_vGr(k_g)>s_rg(p))
                break;
            end
            k_g = k_g + 1;
        end
        
        CDF_r(p) = k_r/N;
        CDF_g(p) = k_g/N;
    end
    
    M = 100;
    dM = (xmax_p-xmin_p)/M;
    x = xmin + ([1:M+1]-1)*dM;
    
    Dmax = zeros(M,1);
    
    for m = 1:M,
        idx_s_rg_m = find((s_rg >= x(m))&(s_rg < x(m+1)));
        if (~isempty(idx_s_rg_m))
            Dmax(m) = max(abs(CDF_g(idx_s_rg_m)-CDF_r(idx_s_rg_m)));
        end
    end
    
    a_critical = 1.63/sqrt(N)*(xmax_p-xmin_p);
    d.KSI = sum(Dmax)*dM;
    d.KSI_r = 100*sum(Dmax)*dM/a_critical;
    
end

if (d.n >= 100)
    
    % Percentiles
    
    %     nb = min(d.n,100);
    %     xmin = min([vR ; vGr]);
    %     xmax = max([vR ; vGr]);
    %     dx = (xmax-xmin)/(nb-1);
    %     vx = xmin + [0:nb-1]*dx;
    %
    %     hcR = zeros(nb,1);
    %     hcG = zeros(nb,1);
    %
    %     for k = 1:nb,
    %         ind = find((vR <= xmin + k*dx));
    %         hcR(k) = length(ind)/d.n;
    %     end
    %
    %     for k = 1:nb,
    %         ind = find((vGr <= xmin + k*dx));
    %         hcG(k) = length(ind)/d.n;
    %     end
    %
    %     %     hRG = zeros(nb,nb);
    %     %     for k1 = 1:nb,
    %     %         for k2 = 1:nb,
    %     %             ind = find((vGr > xmin + (k1-1)*dx)&(vGr <= xmin + k1*dx)&...
    %     %                 (vR > xmin + (k2-1)*dx)&(vR <= xmin + k2*dx));
    %     %             hRG(k1,k2) = length(ind)/d.n;
    %     %         end
    %     %     end
    %
    %     D = abs(hcG-hcR);
    %     KSI = sum(D)*dx;
    %
    %     Vc = 1.63/sqrt(d.n);
    %     a_c = Vc*(xmax-xmin);
    %
    %     d.KSI_r2 = 100*KSI/a_c;
    
    if (exist('prctile')~=0)
        d.p90 = prctile(abs(vGr-vR),90);
        d.p90_r = 100*d.p90/mean_r;
        d.p95 = prctile(abs(vGr-vR),95);
        d.p95_r = 100*d.p95/mean_r;
        d.p99p7 = prctile(abs(vGr-vR),99.7);
        d.p99p7_r = 100*d.p99p7/mean_r;
    end
    
else
    d.KSI_r = nan;
end

% prepare unit to print it
if ~strcmp(unit, '')
    unit_par = [' (' unit ')']; % add space prefix + parenthesis
    unit = [' ' unit]; % add space prefix
else
    unit_par = '';
end
    
if (nargout == 0)
    if reg > 1
        fprintf(1,'TYPE REG: %s (#%d)\n', d.nomr, d.numr);
        fprintf(1,'COEF REG: [%.2f, %.2f]\n', d.r(1), d.r(2));
    end
    fprintf(1,'NDATA:    %d\n', d.n);
    fprintf(1,'MEAN:     %.2f%s\n', d.mean_ref, unit);
    fprintf(1,'Bias:     %0.2f %% (%.2f%s)\n', d.bias_r, d.bias, unit);
    fprintf(1,'MAE:      %0.2f %% (%.2f%s)\n', d.mae_r, d.mae, unit);
    fprintf(1,'STDE:     %0.2f %% (%.2f%s)\n', d.stdev_r, d.stdev, unit);
    fprintf(1,'RMSE:     %0.2f %% (%.2f%s)\n', d.rmse_r, d.rmse, unit);
    if (d.n >= 10)
        fprintf(1,'CC:       %0.3f\n', d.cc);
    end
    if (d.n >= 100)
        fprintf(1,'KSI:      %05.2f %% (%.2f)\n', d.KSI_r, d.KSI);
        %fprintf(1,'P90:      %05.2f %% (%.2f)\n', d.p90_r, d.p90);
        %fprintf(1,'P95:      %05.2f %% (%.2f)\n', d.p95_r, d.p95);
        %fprintf(1,'P99.7:    %05.2f %% (%.2f)\n', d.p99p7_r, d.p99p7);
    end
    fprintf(1,'\n');
end

hsign = inline('char(43+2*(sign(x)==-1))','x');

% display scatter plot
if (graph_scatter>0)
    
    f = gcf; % get current figure or create a new one
    
    % if graph_scatter == 1, draw a comparison graph in the current figure
    if graph_scatter == 1
        if strcmp(get(f,'visible'),'on') % if figure is visible
            figure(f) % bring figure to front
        end
        clf % clear figure
        set(f,'position',[100 50 800 675]) % set position and size
    end
    
    % clim parameter can specify the scaling
    if ~isempty(clim)
        xmin = clim(1);
        xmax = clim(2);
        ymin = clim(1);
        ymax = clim(2);
    else
        % LS 11/2017: compute min and max, excluding outliers
        xmin = percentile100(vR(:),0.1); % min(vR(:));
        xmax = percentile100(vR(:),99.9); % max(vR(:));
        ymin = percentile100(vGr(:),0.1); % min(vGr(:));
        ymax = percentile100(vGr(:),99.9); % max(vGr(:));
    end
    vmin = min(xmin,ymin);
    vmax = max(xmax,ymax);
    dmax = vmax-vmin;
    
    nv = min(round(d.n/20),128);
    nv = max(nv, 10); % make sure nv > 0
    v = linspace(vmin,vmax,nv); % generates nv points between vmin and vmax
    if (exist('histn','file')==2)
        [N,C] = histn([vGr(:), vR(:)], v, v); % computes an histogram for vGr and vR where v defines the histogram buckets
        %         N = 100*N/length(vR);
        %         iN = interp2(C{2},C{1},N,vR, vGr,'lin');
        %         find(isnan(iN));
        %         scatter(vR, vGr, 5^2, iN, 'filled');
        N = 100*N/length(vR);
        ncd = 2048; % EW 06/2019: use 2048 color map (instead of 256) to display more outliers
        vcd = linspace(min(N(N(:)>0)),percentile100(N(:),99.97),ncd); % generates ncd points between N min and max
        dcd = vcd(2)-vcd(1);
        Ncd = round((N-vcd(1))/dcd)+1;
        Ncd(Ncd<1) = 1;
        Ncd(Ncd>ncd) = ncd;
        Ncd(N<eps) = 0;
        NNcd = zeros(size(N));
        NNcd(Ncd~=0) = vcd(Ncd(Ncd~=0));
        NNcd(Ncd==0) = 0;
        map = jet(ncd);
        map(1,:) = [1 1 1];
        colormap(gca, map)
        %         vC2_i = linspace(C{2}(1),C{2}(end),1024);
        %         vC1_i = linspace(C{1}(1),C{1}(end),1024);
        %         [MC2_i,MC1_i] = meshgrid(vC2_i,vC1_i);
        %         NNcd_i = interp2(C{2},C{1},NNcd,MC2_i,MC1_i,'*lin',nan);
        imagesc(C{2},C{1},log10(NNcd)),axis xy
        %         imagesc(vC2_i,vC1_i,log10(NNcd_i)),axis xy
        hold on
        plot([vmin vmax], [vmin vmax], 'k--');
        hold off
        % print color bar
        hc = colorbar;
        yt = get(hc,'Ytick');
        yt_lbl = cell(length(yt),1);
        for k = 1:length(yt),
            % pretty colorbar scale
            temp_lab = round(10^yt(k)/100*d.n);
            yt_lbl{k} = sprintf('%d',temp_lab);
        end
        set(hc,'Yticklabel',yt_lbl);
    else
        hp = plot(vR, vGr, 'r.', ...
            [vmin vmax], [vmin vmax], 'k--');
    end
    hold on
    axis([vmin vmax vmin vmax])
    axis square
    %     legend(hp,sprintf('(ref,estimation+%s)',nom_cr{reg}),'x = y','location','best')
    xlabel([labels{2} unit_par])
    if (d.numr == 1)
        ylabel([labels{1} unit_par])
    elseif (d.numr == 2)|(d.numr == 4)|(d.numr == 5)
        ylabel(sprintf('%.2f * %s %c %.2f%s', d.r(1), labels{1}, hsign(d.r(2)), abs(d.r(2)), unit_par))
    else
        ylabel(sprintf('%.2f * %s%s', d.r(1), labels{1}, unit_par))
    end
    
    % LS 09/2017: avoid crash if vmax-vmin < 0.5
    if vmax-vmin > 0.5
        set(gca,'Xtick',round(linspace(vmin,vmax,5)*10)/10,'Ytick',round(linspace(vmin,vmax,5)*10)/10);
    else
        set(gca,'Xtick',linspace(vmin,vmax,5),'Ytick',linspace(vmin,vmax,5));
    end
    
    if reg==1
        txt_stat{1} = sprintf('NDATA: %d', d.n);
        txt_stat{2} = sprintf('MEAN:  %.2f%s', d.mean_ref, unit);
        txt_stat{3} = sprintf('Bias:  %0.2f %% (%.2f%s)', d.bias_r, d.bias, unit);
        txt_stat{4} = sprintf('MAE:   %0.2f %% (%.2f%s)', d.mae_r, d.mae, unit);
        txt_stat{5} = sprintf('STDE:  %0.2f %% (%.2f%s)', d.stdev_r, d.stdev, unit);
        txt_stat{6} = sprintf('RMSE:  %0.2f %% (%.2f%s)', d.rmse_r, d.rmse, unit);
        if (d.n >= 10)
            txt_stat{7} = sprintf('CC:    %0.3f', d.cc);
        end
    else
        txt_stat{1} = sprintf('COEF REG: [%.2f, %.2f]', d.r(1), d.r(2));
        txt_stat{2} = sprintf('NDATA: %d', d.n);
        txt_stat{3} = sprintf('MEAN:  %.2f%s', d.mean_ref, unit);
        txt_stat{4} = sprintf('Bias:  %0.2f %% (%.2f%s)', d.bias_r, d.bias, unit);
        txt_stat{5} = sprintf('MAE:   %0.2f %% (%.2f%s)', d.mae_r, d.mae, unit);
        txt_stat{6} = sprintf('STDE:  %0.2f %% (%.2f%s)', d.stdev_r, d.stdev, unit);
        txt_stat{7} = sprintf('RMSE:  %0.2f %% (%.2f%s)', d.rmse_r, d.rmse, unit);
        if (d.n >= 10)
            txt_stat{8} = sprintf('CC:    %0.3f', d.cc);
        end
%        if (d.n >= 100)
%         txt_stat{9} = sprintf('KSI:   %05.2f %% (%.2f)', d.KSI_r, d.KSI);
%        end
    end

    xt = get(gca,'Xtick');
    yt = get(gca,'Ytick');
    for kt = 1:length(xt),
       plot([xt(kt) xt(kt)],[yt(1) yt(end)],'k--');
    end
    for kt = 1:length(xt),
       plot([xt(1) xt(end)],[yt(kt) yt(kt)],'k--');
    end
    
    text(vmin+dmax/50,vmax-dmax/50,txt_stat,...
        'VerticalAlignment','top', 'FontName','Courier', 'FontSize',9, 'BackgroundColor',[1 1 1]*0.95);

    hold off

    % force figure drawing now to workaround issues when opening many figures quickly
    refreshdata(f)
    
end
